"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.systemRoutes = systemRoutes;
var _configSchema = require("@kbn/config-schema");
var _app = require("../../common/constants/app");
var _error_wrapper = require("../client/error_wrapper");
var _log = require("../lib/log");
var _capabilities = require("../lib/capabilities");
var _spaces_utils = require("../lib/spaces_utils");
var _node_utils = require("../lib/node_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * System routes
 */
function systemRoutes({
  router,
  mlLicense,
  routeGuard
}, {
  getSpaces,
  cloud,
  resolveMlCapabilities,
  isServerless
}) {
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/_has_privileges`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetMlInfo']
      }
    },
    summary: 'Check privileges',
    description: 'Checks if the user has required privileges'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: _configSchema.schema.maybe(_configSchema.schema.any())
      }
    }
  }, routeGuard.basicLicenseAPIGuard(async ({
    mlClient,
    client,
    request,
    response
  }) => {
    try {
      const {
        asCurrentUser
      } = client;
      let upgradeInProgress = false;
      try {
        const body = await mlClient.info();
        // if ml indices are currently being migrated, upgrade_mode will be set to true
        // pass this back with the privileges to allow for the disabling of UI controls.
        upgradeInProgress = body.upgrade_mode === true;
      } catch (error) {
        // if the ml.info check fails, it could be due to the user having insufficient privileges
        // most likely they do not have the ml_user role and therefore will be blocked from using
        // ML at all. However, we need to catch this error so the privilege check doesn't fail.
        if (error.status === 403) {
          _log.mlLog.info('Unable to determine whether upgrade is being performed due to insufficient user privileges');
        } else {
          _log.mlLog.warn('Unable to determine whether upgrade is being performed');
        }
      }
      if (mlLicense.isSecurityEnabled() === false) {
        // if xpack.security.enabled has been explicitly set to false
        // return that security is disabled and don't call the privilegeCheck endpoint
        return response.ok({
          body: {
            upgradeInProgress
          }
        });
      } else {
        const body = await asCurrentUser.security.hasPrivileges({
          body: request.body
        });
        return response.ok({
          body: {
            hasPrivileges: body,
            upgradeInProgress
          }
        });
      }
    } catch (error) {
      return response.customError((0, _error_wrapper.wrapError)(error));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/ml_capabilities`,
    access: 'internal',
    summary: 'Check ML capabilities',
    description: 'Checks ML capabilities'
  }).addVersion({
    version: '1',
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization because permissions will be checked by elasticsearch'
      }
    },
    validate: false
  }, routeGuard.basicLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const {
        isMlEnabledInSpace
      } = (0, _spaces_utils.spacesUtilsProvider)(getSpaces, request);
      const mlCapabilities = await resolveMlCapabilities(request);
      if (mlCapabilities === null) {
        return response.customError((0, _error_wrapper.wrapError)(new Error('resolveMlCapabilities is not defined')));
      }
      const {
        getCapabilities
      } = (0, _capabilities.capabilitiesProvider)(mlClient, mlCapabilities, mlLicense, isMlEnabledInSpace);
      return response.ok({
        body: await getCapabilities()
      });
    } catch (error) {
      return response.customError((0, _error_wrapper.wrapError)(error));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/ml_node_count`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetMlInfo']
      }
    },
    summary: 'Get the number of ML nodes',
    description: 'Returns the number of ML nodes'
  }).addVersion({
    version: '1',
    validate: false
  }, routeGuard.basicLicenseAPIGuard(async ({
    client,
    response
  }) => {
    try {
      return response.ok({
        body: await (0, _node_utils.getMlNodeCount)(client)
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/info`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetMlInfo']
      }
    },
    summary: 'Get ML info',
    description: 'Returns defaults and limits used by machine learning'
  }).addVersion({
    version: '1',
    validate: false
  }, routeGuard.basicLicenseAPIGuard(async ({
    mlClient,
    response,
    client
  }) => {
    try {
      const body = await mlClient.info();
      const cloudId = cloud === null || cloud === void 0 ? void 0 : cloud.cloudId;
      const isCloudTrial = (cloud === null || cloud === void 0 ? void 0 : cloud.trialEndDate) && Date.now() < cloud.trialEndDate.getTime();
      let isMlAutoscalingEnabled = false;
      try {
        // kibana_system user does not have the manage_autoscaling cluster privilege.
        // perform this check as a current user.
        await client.asCurrentUser.autoscaling.getAutoscalingPolicy({
          name: 'ml'
        });
        isMlAutoscalingEnabled = true;
      } catch (e) {
        // If ml autoscaling policy doesn't exist or the user does not have privileges to fetch it,
        // check the number of lazy ml nodes to determine if autoscaling is enabled.
        const lazyMlNodeCount = await (0, _node_utils.getLazyMlNodeCount)(client);
        isMlAutoscalingEnabled = lazyMlNodeCount > 0;
      }
      return response.ok({
        body: {
          ...body,
          cloudId,
          isCloudTrial,
          cloudUrl: cloud.baseUrl,
          isMlAutoscalingEnabled,
          showNodeInfo: !isServerless,
          showLicenseInfo: !isServerless
        }
      });
    } catch (error) {
      return response.customError((0, _error_wrapper.wrapError)(error));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/es_search`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetJobs']
      }
    },
    summary: 'ES Search wrapper',
    // @ts-expect-error TODO(https://github.com/elastic/kibana/issues/196095): Replace {RouteDeprecationInfo}
    deprecated: true
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: _configSchema.schema.maybe(_configSchema.schema.any())
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    request,
    response
  }) => {
    try {
      const body = await client.asCurrentUser.search(request.body);
      return response.ok({
        body
      });
    } catch (error) {
      return response.customError((0, _error_wrapper.wrapError)(error));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/index_exists`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetFieldInfo']
      }
    },
    summary: 'ES Field caps wrapper checks if index exists'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: _configSchema.schema.object({
          indices: _configSchema.schema.arrayOf(_configSchema.schema.string())
        })
      }
    }
  }, routeGuard.basicLicenseAPIGuard(async ({
    client,
    request,
    response
  }) => {
    try {
      const {
        indices
      } = request.body;
      const results = await Promise.all(indices.map(async index => client.asCurrentUser.indices.exists({
        index,
        allow_no_indices: false
      })));
      const result = indices.reduce((acc, cur, i) => {
        acc[cur] = {
          exists: results[i]
        };
        return acc;
      }, {});
      return response.ok({
        body: result
      });
    } catch (error) {
      return response.customError((0, _error_wrapper.wrapError)(error));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/reindex_with_pipeline`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canCreateTrainedModels']
      }
    },
    summary: 'ES reindex wrapper to reindex with pipeline'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: _configSchema.schema.any()
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    request,
    response
  }) => {
    const reindexRequest = {
      body: request.body,
      // Create a task and return task id instead of blocking until complete
      wait_for_completion: false
    };
    try {
      const result = await client.asCurrentUser.reindex(reindexRequest);
      return response.ok({
        body: result
      });
    } catch (error) {
      return response.customError((0, _error_wrapper.wrapError)(error));
    }
  }));
}