"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.jobRoutes = jobRoutes;
var _configSchema = require("@kbn/config-schema");
var _app = require("../../common/constants/app");
var _error_wrapper = require("../client/error_wrapper");
var _anomaly_detectors_schema = require("./schemas/anomaly_detectors_schema");
var _request_authorization = require("../lib/request_authorization");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Routes for the anomaly detectors
 */
function jobRoutes({
  router,
  routeGuard
}) {
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetJobs']
      }
    },
    summary: 'Gets anomaly detectors',
    description: 'Returns the list of anomaly detection jobs.'
  }).addVersion({
    version: '1',
    validate: {
      response: {
        200: {
          body: _anomaly_detectors_schema.getAnomalyDetectorsResponse,
          description: 'Anomaly detectors response'
        }
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    response
  }) => {
    try {
      const body = await mlClient.getJobs();
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/{jobId}`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetJobs']
      }
    },
    summary: 'Gets anomaly detector by ID',
    description: 'Returns the anomaly detection job by ID'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _anomaly_detectors_schema.jobIdSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const {
        jobId
      } = request.params;
      const body = await mlClient.getJobs({
        job_id: jobId
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/_stats`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetJobs']
      }
    },
    summary: 'Gets anomaly detectors stats',
    description: 'Returns the anomaly detection jobs statistics.'
  }).addVersion({
    version: '1',
    validate: false
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    response
  }) => {
    try {
      const body = await mlClient.getJobStats();
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/{jobId}/_stats`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetJobs']
      }
    },
    summary: 'Gets anomaly detector stats by ID',
    description: 'Returns the anomaly detection job statistics by ID'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _anomaly_detectors_schema.jobIdSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const {
        jobId
      } = request.params;
      const body = await mlClient.getJobStats({
        job_id: jobId
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.put({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/{jobId}`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canCreateJob']
      }
    },
    summary: 'Creates an anomaly detection job',
    description: 'Creates an anomaly detection job.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _anomaly_detectors_schema.jobIdSchema,
        body: _configSchema.schema.object(_anomaly_detectors_schema.anomalyDetectionJobSchema)
      },
      response: {
        200: {
          body: () => _configSchema.schema.any(),
          description: 'The configuration of the job that has been created.'
        }
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const {
        jobId
      } = request.params;
      const body = await mlClient.putJob({
        job_id: jobId,
        // @ts-expect-error job type custom_rules is incorrect
        body: request.body
      }, (0, _request_authorization.getAuthorizationHeader)(request));
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/{jobId}/_update`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canUpdateJob']
      }
    },
    summary: 'Updates an anomaly detection job',
    description: 'Updates certain properties of an anomaly detection job.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _anomaly_detectors_schema.jobIdSchema,
        body: _anomaly_detectors_schema.anomalyDetectionUpdateJobSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const {
        jobId
      } = request.params;
      const body = await mlClient.updateJob({
        job_id: jobId,
        // @ts-expect-error MlDetector is not compatible
        body: request.body
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/{jobId}/_open`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canOpenJob']
      }
    },
    summary: 'Opens an anomaly detection job',
    description: 'Opens an anomaly detection job.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _anomaly_detectors_schema.jobIdSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const {
        jobId
      } = request.params;
      const body = await mlClient.openJob({
        job_id: jobId
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/{jobId}/_close`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canCloseJob']
      }
    },
    summary: 'Closes an anomaly detection job',
    description: 'Closes an anomaly detection job.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _anomaly_detectors_schema.jobIdSchema,
        query: _anomaly_detectors_schema.forceQuerySchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const options = {
        job_id: request.params.jobId
      };
      const force = request.query.force;
      if (force !== undefined) {
        options.force = force;
      }
      const body = await mlClient.closeJob(options);
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.delete({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/{jobId}`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canDeleteJob']
      }
    },
    summary: 'Deletes an anomaly detection job',
    description: 'Deletes specified anomaly detection job.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _anomaly_detectors_schema.jobIdSchema,
        query: _anomaly_detectors_schema.forceQuerySchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const options = {
        job_id: request.params.jobId,
        wait_for_completion: false
      };
      const force = request.query.force;
      if (force !== undefined) {
        options.force = force;
      }
      const body = await mlClient.deleteJob(options);
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.delete({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/{jobId}/_forecast/{forecastId}`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canDeleteForecast']
      }
    },
    summary: 'Deletes specified forecast for specified job',
    description: 'Deletes a specified forecast for the specified anomaly detection job.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _anomaly_detectors_schema.deleteForecastSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const {
        jobId,
        forecastId
      } = request.params;
      const body = await mlClient.deleteForecast({
        job_id: jobId,
        forecast_id: forecastId
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/{jobId}/_forecast`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canForecastJob']
      }
    },
    summary: 'Creates forecast for specified job',
    description: 'Creates a forecast for the specified anomaly detection job, predicting the future behavior of a time series by using its historical behavior.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _anomaly_detectors_schema.jobIdSchema,
        body: _anomaly_detectors_schema.forecastAnomalyDetector
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const jobId = request.params.jobId;
      const body = await mlClient.forecast({
        job_id: jobId,
        body: {
          ...request.body
        }
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/{jobId}/results/buckets/{timestamp?}`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetJobs']
      }
    },
    summary: 'Gets bucket scores',
    description: 'The get buckets API presents a chronological view of the records, grouped by bucket.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _anomaly_detectors_schema.getBucketParamsSchema,
        body: _anomaly_detectors_schema.getBucketsSchema
      },
      response: {
        200: {
          body: () => _configSchema.schema.object({
            count: _configSchema.schema.number(),
            buckets: _configSchema.schema.arrayOf(_configSchema.schema.any())
          })
        }
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const body = await mlClient.getBuckets({
        job_id: request.params.jobId,
        timestamp: request.params.timestamp,
        body: request.body
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/{jobId}/results/overall_buckets`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetJobs']
      }
    },
    summary: 'Get overall buckets',
    description: 'Retrieves overall bucket results that summarize the bucket results of multiple anomaly detection jobs.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _anomaly_detectors_schema.jobIdSchema,
        body: _anomaly_detectors_schema.getOverallBucketsSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      var _request$body$overall;
      const body = await mlClient.getOverallBuckets({
        job_id: request.params.jobId,
        top_n: request.body.topN,
        bucket_span: request.body.bucketSpan,
        start: request.body.start !== undefined ? String(request.body.start) : undefined,
        end: request.body.end !== undefined ? String(request.body.end) : undefined,
        overall_score: (_request$body$overall = request.body.overall_score) !== null && _request$body$overall !== void 0 ? _request$body$overall : 0
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/{jobId}/results/categories/{categoryId}`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetJobs']
      }
    },
    summary: 'Get categories',
    description: 'Retrieves the categories results for the specified job ID and category ID.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _anomaly_detectors_schema.getCategoriesSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const body = await mlClient.getCategories({
        job_id: request.params.jobId,
        category_id: request.params.categoryId
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/{jobId}/model_snapshots`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetJobs']
      }
    },
    summary: 'Get model snapshots by job ID',
    description: 'Returns the model snapshots for the specified job ID'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _anomaly_detectors_schema.jobIdSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const body = await mlClient.getModelSnapshots({
        job_id: request.params.jobId
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/{jobId}/model_snapshots/{snapshotId}`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetJobs']
      }
    },
    summary: 'Get model snapshots by id',
    description: 'Returns the model snapshots for the specified job ID and snapshot ID'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _anomaly_detectors_schema.getModelSnapshotsSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const body = await mlClient.getModelSnapshots({
        job_id: request.params.jobId,
        snapshot_id: request.params.snapshotId
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/{jobId}/model_snapshots/{snapshotId}/_update`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canCreateJob']
      }
    },
    summary: 'Updates model snapshot by snapshot ID',
    description: 'Updates the model snapshot for the specified snapshot ID'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _anomaly_detectors_schema.updateModelSnapshotsSchema,
        body: _anomaly_detectors_schema.updateModelSnapshotBodySchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const body = await mlClient.updateModelSnapshot({
        job_id: request.params.jobId,
        snapshot_id: request.params.snapshotId,
        body: request.body
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.delete({
    path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/{jobId}/model_snapshots/{snapshotId}`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canCreateJob']
      }
    },
    summary: 'Deletes model snapshots by snapshot ID',
    description: 'Deletes the model snapshot for the specified snapshot ID'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _anomaly_detectors_schema.updateModelSnapshotsSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const body = await mlClient.deleteModelSnapshot({
        job_id: request.params.jobId,
        snapshot_id: request.params.snapshotId
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
}