"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initializeSelection = void 0;
var _managementSettingsIds = require("@kbn/management-settings-ids");
var _datasets = require("../../../../../common/datasets");
var _data_source_selection = require("../../../../../common/data_source_selection");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const initializeSelection = ({
  datasetsClient,
  dataViews,
  events,
  uiSettings
}) => context => async send => {
  /**
   * First validation.
   * The selection is a data view.
   */
  if ((0, _data_source_selection.isDataViewSelection)(context.dataSourceSelection)) {
    let dataViewSelection = context.dataSourceSelection;
    /**
     * If the selection is unresolved, perform a look up to retrieve it.
     */
    if (dataViewSelection.selection.dataView.isUnresolvedDataType()) {
      try {
        dataViewSelection = await lookupUnresolvedDataViewSelection(dataViewSelection, {
          dataViews
        });
        if (dataViewSelection === null) {
          return send('DATAVIEW_SELECTION_RESTORE_FAILURE');
        }
      } catch {
        return send('DATAVIEW_SELECTION_RESTORE_FAILURE');
      }
    }

    /**
     * If the selection is a data view which is not of logs type, invoke the customization event for unknown data views.
     */
    if (!dataViewSelection.selection.dataView.testAgainstAllowedList(uiSettings.get(_managementSettingsIds.OBSERVABILITY_LOGS_EXPLORER_ALLOWED_DATA_VIEWS_ID)) && events !== null && events !== void 0 && events.onUknownDataViewSelection) {
      return events.onUknownDataViewSelection(context);
    }
    return send({
      type: 'INITIALIZE_DATA_VIEW',
      data: dataViewSelection
    });
  }

  /**
   * Second validation.
   * If the selection is an unresolved dataset, perform a look up against integrations.
   */
  if ((0, _data_source_selection.isUnresolvedDatasetSelection)(context.dataSourceSelection)) {
    try {
      const datasetSelection = await lookupUnresolvedDatasetSelection(context.dataSourceSelection, {
        datasetsClient
      });
      if (datasetSelection !== null) {
        return send({
          type: 'INITIALIZE_DATASET',
          data: datasetSelection
        });
      }
    } catch {
      return send('DATASET_SELECTION_RESTORE_FAILURE');
    }
  }

  /**
   * For any remaining case, initialize the current dataset selection
   */
  return send('INITIALIZE_DATASET');
};
exports.initializeSelection = initializeSelection;
const lookupUnresolvedDatasetSelection = async (unresolvedDatasetSelection, {
  datasetsClient
}) => {
  var _unresolvedDatasetSel;
  const nameQuery = (_unresolvedDatasetSel = unresolvedDatasetSelection.selection.dataset.parentIntegration) === null || _unresolvedDatasetSel === void 0 ? void 0 : _unresolvedDatasetSel.name;
  if (nameQuery) {
    return null;
  }
  const {
    items
  } = await datasetsClient.findIntegrations({
    nameQuery
  });

  // There should only be one matching integration with the given name
  // If no integration matches, skip the update and listen for user changes
  const installedIntegration = items[0];
  if (!installedIntegration) {
    return null;
  }

  // If no dataset matches the passed name for the retrieved integration,
  // skip the update and listen for user changes
  const targetDataset = installedIntegration.datasets.find(d => d.name === nameQuery);
  if (!targetDataset) {
    return null;
  }
  const dataset = _datasets.Dataset.create(targetDataset, installedIntegration);
  return _data_source_selection.SingleDatasetSelection.create(dataset);
};
const lookupUnresolvedDataViewSelection = async (unresolvedDataViewSelection, {
  dataViews
}) => {
  var _resolvedDataView$id;
  const resolvedDataView = await dataViews.get(unresolvedDataViewSelection.toDataviewSpec().id);
  if (!resolvedDataView) {
    return null;
  }
  return _data_source_selection.DataViewSelection.fromSelection({
    dataView: {
      id: (_resolvedDataView$id = resolvedDataView.id) !== null && _resolvedDataView$id !== void 0 ? _resolvedDataView$id : '',
      kibanaSpaces: resolvedDataView.namespaces,
      name: resolvedDataView.name,
      title: resolvedDataView.getIndexPattern(),
      type: resolvedDataView.type
    }
  });
};