"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSetInitialValue = exports.readLoadFromParam = void 0;
var _lodash = require("lodash");
var _queryString = require("query-string");
var _lzString = require("lz-string");
var _i18n = require("@kbn/i18n");
var _react = require("react");
var _constants = require("../../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Util function for reading the load_from parameter from the current url.
 */

const readLoadFromParam = () => {
  const [, queryString] = (window.location.hash || window.location.search || '').split('?');
  const queryParams = (0, _queryString.parse)(queryString || '', {
    sort: false
  });
  return queryParams.load_from;
};

/**
 * Hook that sets the initial value in the Console editor.
 *
 * @param params The {@link SetInitialValueParams} to use.
 */
exports.readLoadFromParam = readLoadFromParam;
const useSetInitialValue = params => {
  const {
    localStorageValue,
    setValue,
    toasts
  } = params;
  const isInitialValueSet = (0, _react.useRef)(false);
  (0, _react.useEffect)(() => {
    const ALLOWED_PATHS = ['/guide/', '/docs/'];
    const loadBufferFromRemote = async url => {
      if (/^https?:\/\//.test(url)) {
        // Check if this is a valid URL
        try {
          new URL(url);
        } catch (e) {
          return;
        }
        // Parse the URL to avoid issues with spaces and other special characters.
        const parsedURL = new URL(url);
        // Validate protocol, hostname, and allowed path to prevent request forgery
        if (parsedURL.protocol === 'https:' && parsedURL.hostname === 'www.elastic.co' && ALLOWED_PATHS.some(path => parsedURL.pathname.startsWith(path))) {
          const resp = await fetch(parsedURL);
          const data = await resp.text();
          setValue(`${localStorageValue !== null && localStorageValue !== void 0 ? localStorageValue : ''}\n\n${data}`);
        } else {
          toasts.addWarning(_i18n.i18n.translate('console.monaco.loadFromDataUnrecognizedUrlErrorMessage', {
            defaultMessage: 'Only URLs with the Elastic domain (www.elastic.co) can be loaded in Console.'
          }));
        }
      }

      // If we have a data URI instead of HTTP, LZ-decode it. This enables
      // opening requests in Console from anywhere in Kibana.
      if (/^data:/.test(url)) {
        const data = (0, _lzString.decompressFromEncodedURIComponent)(url.replace(/^data:text\/plain,/, ''));

        // Show a toast if we have a failure
        if (data === null || data === '') {
          toasts.addWarning(_i18n.i18n.translate('console.monaco.loadFromDataUriErrorMessage', {
            defaultMessage: 'Unable to load data from the load_from query parameter in the URL'
          }));
          return;
        }
        setValue(data);
      }
    };

    // Support for loading a console snippet from a remote source, like support docs.
    const onHashChange = (0, _lodash.debounce)(async () => {
      const url = readLoadFromParam();
      if (!url) {
        return;
      }
      await loadBufferFromRemote(url);
    }, 200);
    window.addEventListener('hashchange', onHashChange);
    const loadFromParam = readLoadFromParam();

    // Only set the value in the editor if an initial value hasn't been set yet
    if (!isInitialValueSet.current) {
      if (loadFromParam) {
        loadBufferFromRemote(loadFromParam);
      } else {
        // Only set to default input value if the localstorage value is undefined
        setValue(localStorageValue !== null && localStorageValue !== void 0 ? localStorageValue : _constants.DEFAULT_INPUT_VALUE);
      }
      isInitialValueSet.current = true;
    }
    return () => {
      window.removeEventListener('hashchange', onHashChange);
    };
  }, [localStorageValue, setValue, toasts]);
};
exports.useSetInitialValue = useSetInitialValue;