"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSchemaFields = void 0;
var _i18n = require("@kbn/i18n");
var _streamsSchema = require("@kbn/streams-schema");
var _lodash = require("lodash");
var _react = require("react");
var _reactHooks = require("@kbn/react-hooks");
var _use_streams_app_fetch = require("../../../../hooks/use_streams_app_fetch");
var _use_kibana = require("../../../../hooks/use_kibana");
var _types = require("../types");
var _utils = require("../utils");
var _errors = require("../../../../util/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useSchemaFields = ({
  definition,
  refreshDefinition
}) => {
  const {
    dependencies: {
      start: {
        streams: {
          streamsRepositoryClient
        }
      }
    },
    core: {
      notifications: {
        toasts
      }
    }
  } = (0, _use_kibana.useKibana)();
  const abortController = (0, _reactHooks.useAbortController)();
  const {
    value: unmappedFieldsValue,
    loading: isLoadingUnmappedFields,
    refresh: refreshUnmappedFields
  } = (0, _use_streams_app_fetch.useStreamsAppFetch)(({
    signal
  }) => {
    return streamsRepositoryClient.fetch('GET /internal/streams/{name}/schema/unmapped_fields', {
      signal,
      params: {
        path: {
          name: definition.stream.name
        }
      }
    });
  }, [definition.stream.name, streamsRepositoryClient]);
  const fields = (0, _react.useMemo)(() => {
    var _unmappedFieldsValue$;
    const inheritedFields = Object.entries(definition.inherited_fields).map(([name, field]) => ({
      name,
      type: field.type,
      format: 'format' in field ? field.format : undefined,
      additionalParameters: (0, _streamsSchema.getAdvancedParameters)(name, field),
      parent: field.from,
      alias_for: field.alias_for,
      status: 'inherited'
    }));
    const mappedFields = Object.entries(definition.stream.ingest.wired.fields).map(([name, field]) => ({
      name,
      type: field.type,
      format: 'format' in field ? field.format : undefined,
      additionalParameters: (0, _streamsSchema.getAdvancedParameters)(name, field),
      parent: definition.stream.name,
      status: 'mapped'
    }));
    const unmappedFields = (_unmappedFieldsValue$ = unmappedFieldsValue === null || unmappedFieldsValue === void 0 ? void 0 : unmappedFieldsValue.unmappedFields.map(field => ({
      name: field,
      parent: definition.stream.name,
      status: 'unmapped'
    }))) !== null && _unmappedFieldsValue$ !== void 0 ? _unmappedFieldsValue$ : [];
    return [...inheritedFields, ...mappedFields, ...unmappedFields];
  }, [definition, unmappedFieldsValue]);
  const refreshFields = (0, _react.useCallback)(() => {
    refreshDefinition();
    refreshUnmappedFields();
  }, [refreshDefinition, refreshUnmappedFields]);
  const updateField = (0, _react.useCallback)(async field => {
    try {
      if (!(0, _types.isSchemaFieldTyped)(field)) {
        throw new Error('The field is not complete or fully mapped.');
      }
      const nextFieldDefinitionConfig = (0, _utils.convertToFieldDefinitionConfig)(field);
      const persistedFieldDefinitionConfig = definition.stream.ingest.wired.fields[field.name];
      if (!hasChanges(persistedFieldDefinitionConfig, nextFieldDefinitionConfig)) {
        throw new Error('The field is not different, hence updating is not necessary.');
      }
      await streamsRepositoryClient.fetch(`PUT /api/streams/{name}/_ingest 2023-10-31`, {
        signal: abortController.signal,
        params: {
          path: {
            name: definition.stream.name
          },
          body: {
            ingest: {
              ...definition.stream.ingest,
              wired: {
                ...definition.stream.ingest.wired,
                fields: {
                  ...definition.stream.ingest.wired.fields,
                  [field.name]: nextFieldDefinitionConfig
                }
              }
            }
          }
        }
      });
      toasts.addSuccess(_i18n.i18n.translate('xpack.streams.streamDetailSchemaEditorEditSuccessToast', {
        defaultMessage: '{field} was successfully edited',
        values: {
          field: field.name
        }
      }));
      refreshFields();
    } catch (error) {
      toasts.addError(new Error(error.body.message), {
        title: _i18n.i18n.translate('xpack.streams.streamDetailSchemaEditorEditErrorToast', {
          defaultMessage: 'Something went wrong editing the {field} field',
          values: {
            field: field.name
          }
        }),
        toastMessage: (0, _errors.getFormattedError)(error).message,
        toastLifeTimeMs: 5000
      });
    }
  }, [abortController.signal, definition, refreshFields, streamsRepositoryClient, toasts]);
  const unmapField = (0, _react.useCallback)(async fieldName => {
    try {
      const persistedFieldDefinitionConfig = definition.stream.ingest.wired.fields[fieldName];
      if (!persistedFieldDefinitionConfig) {
        throw new Error('The field is not mapped, hence it cannot be unmapped.');
      }
      await streamsRepositoryClient.fetch(`PUT /api/streams/{name}/_ingest 2023-10-31`, {
        signal: abortController.signal,
        params: {
          path: {
            name: definition.stream.name
          },
          body: {
            ingest: {
              ...definition.stream.ingest,
              wired: {
                ...definition.stream.ingest.wired,
                fields: (0, _lodash.omit)(definition.stream.ingest.wired.fields, fieldName)
              }
            }
          }
        }
      });
      toasts.addSuccess(_i18n.i18n.translate('xpack.streams.streamDetailSchemaEditorUnmapSuccessToast', {
        defaultMessage: '{field} was successfully unmapped',
        values: {
          field: fieldName
        }
      }));
      refreshFields();
    } catch (error) {
      toasts.addError(error, {
        title: _i18n.i18n.translate('xpack.streams.streamDetailSchemaEditorUnmapErrorToast', {
          defaultMessage: 'Something went wrong unmapping the {field} field',
          values: {
            field: fieldName
          }
        }),
        toastMessage: (0, _errors.getFormattedError)(error).message,
        toastLifeTimeMs: 5000
      });
    }
  }, [abortController.signal, definition, refreshFields, streamsRepositoryClient, toasts]);
  return {
    fields,
    isLoadingUnmappedFields,
    refreshFields,
    unmapField,
    updateField
  };
};
exports.useSchemaFields = useSchemaFields;
const hasChanges = (field, fieldUpdate) => {
  return !(0, _lodash.isEqual)(field, fieldUpdate);
};