"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GetSLOGroupings = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_SIZE = 100;
class GetSLOGroupings {
  constructor(definitionClient, esClient, sloSettings, spaceId) {
    this.definitionClient = definitionClient;
    this.esClient = esClient;
    this.sloSettings = sloSettings;
    this.spaceId = spaceId;
  }
  async execute(sloId, params) {
    var _params$instanceId$sp, _response$aggregation, _response$aggregation2, _response$aggregation3, _params$size, _response$aggregation4;
    const {
      slo
    } = await this.definitionClient.execute(sloId, this.spaceId, params.remoteName);
    const groupingKeys = [slo.groupBy].flat();
    if (groupingKeys.includes(_sloSchema.ALL_VALUE) || params.instanceId === _sloSchema.ALL_VALUE) {
      throw new Error('Ungrouped SLO cannot be queried for available groupings');
    }
    if (!groupingKeys.includes(params.groupingKey)) {
      throw new Error("Provided groupingKey doesn't match the SLO's groupBy field");
    }
    const groupingValues = (_params$instanceId$sp = params.instanceId.split(',')) !== null && _params$instanceId$sp !== void 0 ? _params$instanceId$sp : [];
    if (groupingKeys.length !== groupingValues.length) {
      throw new Error('Provided instanceId does not match the number of grouping keys');
    }
    const response = await this.esClient.search({
      index: params.remoteName ? `${params.remoteName}:${_constants.SUMMARY_DESTINATION_INDEX_NAME}` : _constants.SUMMARY_DESTINATION_INDEX_NAME,
      ...generateQuery(slo, params, this.sloSettings)
    });
    return {
      groupingKey: params.groupingKey,
      values: (_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.groupingValues.buckets.map(bucket => bucket.key.value)) !== null && _response$aggregation !== void 0 ? _response$aggregation : [],
      afterKey: ((_response$aggregation3 = response.aggregations) === null || _response$aggregation3 === void 0 ? void 0 : _response$aggregation3.groupingValues.buckets.length) === Number((_params$size = params.size) !== null && _params$size !== void 0 ? _params$size : DEFAULT_SIZE) ? (_response$aggregation4 = response.aggregations) === null || _response$aggregation4 === void 0 ? void 0 : _response$aggregation4.groupingValues.after_key.value : undefined
    };
  }
}
exports.GetSLOGroupings = GetSLOGroupings;
function generateQuery(slo, params, settings) {
  var _params$instanceId$sp2;
  const groupingKeys = [slo.groupBy].flat();
  const groupingValues = (_params$instanceId$sp2 = params.instanceId.split(',')) !== null && _params$instanceId$sp2 !== void 0 ? _params$instanceId$sp2 : [];
  const groupingKeyValuePairs = groupingKeys.map((groupingKey, index) => [groupingKey, groupingValues[index]]);
  const aggs = generateAggs(params);
  const query = {
    size: 0,
    query: {
      bool: {
        filter: [{
          term: {
            'slo.id': slo.id
          }
        }, {
          term: {
            'slo.revision': slo.revision
          }
        },
        // exclude stale summary documents if specified
        ...(!!params.excludeStale ? [{
          range: {
            summaryUpdatedAt: {
              gte: `now-${settings.staleThresholdInHours}h`
            }
          }
        }] : []),
        // Set other groupings as term filters
        ...groupingKeyValuePairs.filter(([groupingKey]) => groupingKey !== params.groupingKey).map(([groupingKey, groupingValue]) => ({
          term: {
            [`slo.groupings.${groupingKey}`]: groupingValue
          }
        })),
        // search on the specified groupingKey
        ...(params.search ? [{
          query_string: {
            default_field: `slo.groupings.${params.groupingKey}`,
            query: `*${params.search.replace(/^\*/, '').replace(/\*$/, '')}*`
          }
        }] : [])]
      }
    },
    aggs
  };
  return query;
}
function generateAggs(params) {
  var _params$size2;
  return {
    groupingValues: {
      composite: {
        size: Number((_params$size2 = params.size) !== null && _params$size2 !== void 0 ? _params$size2 : DEFAULT_SIZE),
        sources: [{
          value: {
            terms: {
              field: `slo.groupings.${params.groupingKey}`
            }
          }
        }],
        ...(params.afterKey ? {
          after: {
            value: params.afterKey
          }
        } : {})
      }
    }
  };
}