"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerConnectorsRoutes = void 0;
var _configSchema = require("@kbn/config-schema");
var _searchConnectors = require("@kbn/search-connectors");
var _common = require("../../common");
var _error_handler = require("../utils/error_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerConnectorsRoutes = ({
  logger,
  http,
  router
}) => {
  router.get({
    path: '/internal/serverless_search/connectors',
    validate: {}
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const privileges = await client.asCurrentUser.security.hasPrivileges({
      index: [{
        names: ['.elastic-connectors'],
        privileges: ['read', 'write']
      }]
    });
    const canManageConnectors = privileges.index['.elastic-connectors'].write;
    const canReadConnectors = privileges.index['.elastic-connectors'].read;
    const connectors = canReadConnectors ? await (0, _searchConnectors.fetchConnectors)(client.asCurrentUser) : [];
    return response.ok({
      body: {
        connectors,
        canManageConnectors,
        canReadConnectors
      }
    });
  }));
  router.get({
    path: '/internal/serverless_search/connector/{connectorId}',
    validate: {
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const connector = await (0, _searchConnectors.fetchConnectorById)(client.asCurrentUser, request.params.connectorId);
    return connector ? response.ok({
      body: {
        connector
      },
      headers: {
        'content-type': 'application/json'
      }
    }) : response.notFound();
  }));
  router.post({
    path: '/internal/serverless_search/connectors',
    validate: {}
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const defaultPipeline = {
      name: _common.DEFAULT_INGESTION_PIPELINE,
      extract_binary_content: true,
      reduce_whitespace: true,
      run_ml_inference: true
    };
    const connector = await (0, _searchConnectors.createConnector)(client.asCurrentUser, {
      indexName: null,
      isNative: false,
      language: null,
      pipeline: defaultPipeline
    });
    return response.ok({
      body: {
        connector
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.post({
    path: '/internal/serverless_search/connectors/{connectorId}/name',
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const result = await (0, _searchConnectors.updateConnectorNameAndDescription)(client.asCurrentUser, request.params.connectorId, {
      name: request.body.name
    });
    return response.ok({
      body: {
        result
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.post({
    path: '/internal/serverless_search/connectors/{connectorId}/description',
    validate: {
      body: _configSchema.schema.object({
        description: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const result = await (0, _searchConnectors.updateConnectorNameAndDescription)(client.asCurrentUser, request.params.connectorId, {
      description: request.body.description
    });
    return response.ok({
      body: {
        result
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.post({
    path: '/internal/serverless_search/connectors/{connectorId}/index_name',
    validate: {
      body: _configSchema.schema.object({
        index_name: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    try {
      const result = await (0, _searchConnectors.updateConnectorIndexName)(client.asCurrentUser, request.params.connectorId, request.body.index_name);
      return response.ok({
        body: {
          result
        },
        headers: {
          'content-type': 'application/json'
        }
      });
    } catch (e) {
      return response.conflict({
        body: e
      });
    }
  }));
  router.post({
    path: '/internal/serverless_search/connectors/{connectorId}/service_type',
    validate: {
      body: _configSchema.schema.object({
        service_type: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const result = await (0, _searchConnectors.updateConnectorServiceType)(client.asCurrentUser, request.params.connectorId, request.body.service_type);
    return response.ok({
      body: {
        result
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.delete({
    path: '/internal/serverless_search/connectors/{connectorId}',
    validate: {
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const result = await (0, _searchConnectors.deleteConnectorById)(client.asCurrentUser, request.params.connectorId);
    return response.ok({
      body: {
        result
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.post({
    path: '/internal/serverless_search/connectors/{connectorId}/configuration',
    validate: {
      body: _configSchema.schema.object({
        configuration: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.number(), _configSchema.schema.boolean()]))
      }),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const result = await (0, _searchConnectors.updateConnectorConfiguration)(client.asCurrentUser, request.params.connectorId, request.body.configuration);
    return response.ok({
      body: result,
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.post({
    path: '/internal/serverless_search/connectors/{connectorId}/sync',
    validate: {
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const result = await (0, _searchConnectors.startConnectorSync)(client.asCurrentUser, {
      connectorId: request.params.connectorId
    });
    return response.ok({
      body: result,
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.get({
    path: '/internal/serverless_search/connectors/{connectorId}/sync_jobs',
    validate: {
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        from: _configSchema.schema.maybe(_configSchema.schema.number()),
        size: _configSchema.schema.maybe(_configSchema.schema.number()),
        type: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const result = await (0, _searchConnectors.fetchSyncJobs)(client.asCurrentUser, request.params.connectorId, request.query.from || 0, request.query.size || 20, request.query.type || 'all');
    return response.ok({
      body: result,
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.post({
    path: '/internal/serverless_search/connectors/{connectorId}/scheduling',
    validate: {
      body: _configSchema.schema.object({
        access_control: _configSchema.schema.object({
          enabled: _configSchema.schema.boolean(),
          interval: _configSchema.schema.string()
        }),
        full: _configSchema.schema.object({
          enabled: _configSchema.schema.boolean(),
          interval: _configSchema.schema.string()
        }),
        incremental: _configSchema.schema.object({
          enabled: _configSchema.schema.boolean(),
          interval: _configSchema.schema.string()
        })
      }),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    await (0, _searchConnectors.updateConnectorScheduling)(client.asCurrentUser, request.params.connectorId, request.body);
    return response.ok();
  }));
};
exports.registerConnectorsRoutes = registerConnectorsRoutes;