"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RuleMigrationsDataMigrationClient = void 0;
var _uuid = require("uuid");
var _rule_migrations_data_base_client = require("./rule_migrations_data_base_client");
var _utils = require("./utils");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RuleMigrationsDataMigrationClient extends _rule_migrations_data_base_client.RuleMigrationsDataBaseClient {
  async create(name) {
    const migrationId = (0, _uuid.v4)();
    const index = await this.getIndexName();
    const profileUid = await this.getProfileUid();
    const createdAt = new Date().toISOString();
    await this.esClient.create({
      refresh: 'wait_for',
      id: migrationId,
      index,
      document: {
        created_by: profileUid,
        created_at: createdAt,
        name
      }
    }).catch(error => {
      this.logger.error(`Error creating migration ${migrationId}: ${error}`);
      throw error;
    });
    return migrationId;
  }

  /**
   *
   * Gets the migration document by id or returns undefined if it does not exist.
   *
   * */
  async get({
    id
  }) {
    const index = await this.getIndexName();
    return this.esClient.get({
      index,
      id
    }).then(this.processHit).catch(error => {
      if ((0, _utils.isNotFoundError)(error)) {
        return undefined;
      }
      this.logger.error(`Error getting migration ${id}: ${error}`);
      throw error;
    });
  }

  /**
   * Gets all migrations from the index.
   */
  async getAll() {
    const index = await this.getIndexName();
    return this.esClient.search({
      index,
      size: _constants.MAX_ES_SEARCH_SIZE,
      // Adjust size as needed
      query: {
        match_all: {}
      },
      _source: true
    }).then(result => this.processResponseHits(result)).catch(error => {
      this.logger.error(`Error getting all migrations:- ${error}`);
      throw error;
    });
  }

  /**
   *
   * Prepares bulk ES delete operation for a migration document based on its id.
   *
   */
  async prepareDelete({
    id
  }) {
    const index = await this.getIndexName();
    const migrationDeleteOperation = {
      delete: {
        _index: index,
        _id: id
      }
    };
    return [migrationDeleteOperation];
  }

  /**
   * Saves a migration as started, updating the last execution parameters with the current timestamp.
   */
  async saveAsStarted({
    id,
    connectorId,
    skipPrebuiltRulesMatching = false
  }) {
    await this.updateLastExecution(id, {
      started_at: new Date().toISOString(),
      connector_id: connectorId,
      is_stopped: false,
      error: null,
      finished_at: null,
      skip_prebuilt_rules_matching: skipPrebuiltRulesMatching
    });
  }

  /**
   * Saves a migration as ended, updating the last execution parameters with the current timestamp.
   */
  async saveAsFinished({
    id
  }) {
    await this.updateLastExecution(id, {
      finished_at: new Date().toISOString()
    });
  }

  /**
   * Saves a migration as failed, updating the last execution parameters with the provided error message.
   */
  async saveAsFailed({
    id,
    error
  }) {
    await this.updateLastExecution(id, {
      error,
      finished_at: new Date().toISOString()
    });
  }

  /**
   * Sets `is_stopped` flag for migration document.
   * It does not update `finished_at` timestamp, `saveAsFinished` or `saveAsFailed` should be called separately.
   */
  async setIsStopped({
    id
  }) {
    await this.updateLastExecution(id, {
      is_stopped: true
    });
  }

  /**
   * Updates the last execution parameters for a migration document.
   */
  async updateLastExecution(id, lastExecutionParams) {
    const index = await this.getIndexName();
    const doc = {
      last_execution: lastExecutionParams
    };
    await this.esClient.update({
      index,
      id,
      refresh: 'wait_for',
      doc,
      retry_on_conflict: 1
    }).catch(error => {
      this.logger.error(`Error updating last execution for migration ${id}: ${error}`);
      throw error;
    });
  }

  /**
   * Updates the migration document with the provided values.
   */
  async update(id, doc) {
    const index = await this.getIndexName();
    await this.esClient.update({
      index,
      id,
      doc,
      refresh: 'wait_for',
      retry_on_conflict: 1
    }).catch(error => {
      this.logger.error(`Error updating migration: ${error}`);
      throw error;
    });
  }
}
exports.RuleMigrationsDataMigrationClient = RuleMigrationsDataMigrationClient;