"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.assetCriticalityPublicDeleteRoute = void 0;
var _utils = require("@kbn/lists-plugin/server/routes/utils");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _zodHelpers = require("@kbn/zod-helpers");
var _entity_analytics = require("../../../../../common/api/entity_analytics");
var _constants = require("../../../../../common/constants");
var _check_and_init_asset_criticality_resources = require("../check_and_init_asset_criticality_resources");
var _audit = require("../audit");
var _audit2 = require("../../audit");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const assetCriticalityPublicDeleteRoute = (router, logger) => {
  router.versioned.delete({
    access: 'public',
    path: _constants.ASSET_CRITICALITY_PUBLIC_URL,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution', `${_constants.APP_ID}-entity-analytics`]
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {
        query: (0, _zodHelpers.buildRouteValidationWithZod)(_entity_analytics.DeleteAssetCriticalityRecordRequestQuery)
      }
    }
  }, async (context, request, response) => {
    var _securitySolution$get;
    const securitySolution = await context.securitySolution;
    (_securitySolution$get = securitySolution.getAuditLogger()) === null || _securitySolution$get === void 0 ? void 0 : _securitySolution$get.log({
      message: 'User attempted to un-assign asset criticality from an entity',
      event: {
        action: _audit.AssetCriticalityAuditActions.ASSET_CRITICALITY_UNASSIGN,
        category: _audit2.AUDIT_CATEGORY.DATABASE,
        type: _audit2.AUDIT_TYPE.DELETION,
        outcome: _audit2.AUDIT_OUTCOME.UNKNOWN
      }
    });
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      await (0, _check_and_init_asset_criticality_resources.checkAndInitAssetCriticalityResources)(context, logger);
      const assetCriticalityClient = securitySolution.getAssetCriticalityDataClient();
      const deletedRecord = await assetCriticalityClient.delete({
        idField: request.query.id_field,
        idValue: request.query.id_value
      }, request.query.refresh);
      return response.ok({
        body: {
          deleted: deletedRecord !== undefined,
          record: deletedRecord
        }
      });
    } catch (e) {
      const error = (0, _securitysolutionEsUtils.transformError)(e);
      return siemResponse.error({
        statusCode: error.statusCode,
        body: {
          message: error.message,
          full_error: JSON.stringify(e)
        },
        bypassErrorFormat: true
      });
    }
  });
};
exports.assetCriticalityPublicDeleteRoute = assetCriticalityPublicDeleteRoute;