"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bootstrapPrebuiltRulesHandler = void 0;
var _keys = require("@kbn/security-solution-features/keys");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _install_ai_prompts = require("../../logic/integrations/install_ai_prompts");
var _utils = require("../../../routes/utils");
var _install_endpoint_package = require("../../logic/integrations/install_endpoint_package");
var _install_prebuilt_rules_package = require("../../logic/integrations/install_prebuilt_rules_package");
var _install_promotion_rules = require("../../logic/integrations/install_promotion_rules");
var _prebuilt_rule_assets_client = require("../../logic/rule_assets/prebuilt_rule_assets_client");
var _prebuilt_rule_objects_client = require("../../logic/rule_objects/prebuilt_rule_objects_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bootstrapPrebuiltRulesHandler = async (context, _, response, logger) => {
  const siemResponse = (0, _utils.buildSiemResponse)(response);
  try {
    const ctx = await context.resolve(['securitySolution', 'alerting', 'core']);
    const securityContext = ctx.securitySolution;
    const savedObjectsClient = ctx.core.savedObjects.client;
    const detectionRulesClient = securityContext.getDetectionRulesClient();
    const ruleAssetsClient = (0, _prebuilt_rule_assets_client.createPrebuiltRuleAssetsClient)(savedObjectsClient);
    const rulesClient = await ctx.alerting.getRulesClient();
    const ruleObjectsClient = (0, _prebuilt_rule_objects_client.createPrebuiltRuleObjectsClient)(rulesClient);
    const productFeatureService = securityContext.getProductFeatureService();
    const isExternalDetectionsEnabled = productFeatureService.isEnabled(_keys.ProductFeatureSecurityKey.externalDetections);
    const packageResults = [];

    // Install packages sequentially to avoid high memory usage
    const prebuiltRulesResult = await (0, _install_prebuilt_rules_package.installPrebuiltRulesPackage)(securityContext, logger);
    packageResults.push({
      name: prebuiltRulesResult.package.name,
      version: prebuiltRulesResult.package.version,
      status: prebuiltRulesResult.status
    });
    let ruleResults;
    if (isExternalDetectionsEnabled) {
      ruleResults = await (0, _install_promotion_rules.installPromotionRules)({
        rulesClient,
        detectionRulesClient,
        ruleAssetsClient,
        ruleObjectsClient,
        fleetServices: securityContext.getInternalFleetServices(),
        logger
      });
    } else {
      const endpointResult = await (0, _install_endpoint_package.installEndpointPackage)(securityContext, logger);
      packageResults.push({
        name: endpointResult.package.name,
        version: endpointResult.package.version,
        status: endpointResult.status
      });
    }
    const responseBody = {
      packages: packageResults,
      rules: ruleResults
    };
    const securityAiPromptsResult = await (0, _install_ai_prompts.installSecurityAiPromptsPackage)(securityContext, logger);
    if (securityAiPromptsResult !== null) {
      responseBody.packages.push({
        name: securityAiPromptsResult.package.name,
        version: securityAiPromptsResult.package.version,
        status: securityAiPromptsResult.status
      });
    }
    logger.debug(`bootstrapPrebuiltRulesHandler: Total packages installed: ${packageResults.length}`);
    return response.ok({
      body: responseBody
    });
  } catch (err) {
    logger.error(`bootstrapPrebuiltRulesHandler: Caught error:`, err);
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    return siemResponse.error({
      body: error.message,
      statusCode: error.statusCode
    });
  }
};
exports.bootstrapPrebuiltRulesHandler = bootstrapPrebuiltRulesHandler;