"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EndpointFleetServicesFactory = void 0;
var _server = require("@kbn/fleet-plugin/server");
var _common = require("@kbn/fleet-plugin/common");
var _errors = require("@kbn/fleet-plugin/server/errors");
var _stringify = require("../../utils/stringify");
var _errors2 = require("../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * The set of Fleet services used by Endpoint
 */

/**
 * Provides centralized way to get all services for Fleet and access internal saved object clients
 */
class EndpointFleetServicesFactory {
  constructor(fleetDependencies, savedObjects, logger) {
    this.fleetDependencies = fleetDependencies;
    this.savedObjects = savedObjects;
    this.logger = logger;
  }
  asInternalUser(spaceId) {
    const {
      agentPolicyService: agentPolicy,
      packagePolicyService: packagePolicy,
      agentService,
      packageService
    } = this.fleetDependencies;
    const agent = spaceId ? agentService.asInternalScopedUser(spaceId) : agentService.asInternalUser;

    // Lazily Initialized at the time it is needed
    let soClient;
    const ensureInCurrentSpace = async ({
      integrationPolicyIds = [],
      agentPolicyIds = [],
      agentIds = []
    }) => {
      if (!soClient) {
        soClient = this.savedObjects.createInternalScopedSoClient({
          spaceId
        });
      }
      return checkInCurrentSpace({
        soClient,
        agentService: agent,
        agentPolicyService: agentPolicy,
        packagePolicyService: packagePolicy,
        integrationPolicyIds,
        agentPolicyIds,
        agentIds
      });
    };
    const getPolicyNamespace = async options => {
      if (!soClient) {
        soClient = this.savedObjects.createInternalScopedSoClient({
          spaceId
        });
      }
      return fetchEndpointPolicyNamespace({
        ...options,
        soClient,
        logger: this.logger,
        packagePolicyService: packagePolicy,
        agentPolicyService: agentPolicy
      });
    };
    return {
      agent,
      agentPolicy,
      packages: packageService.asInternalUser,
      packagePolicy,
      savedObjects: this.savedObjects,
      endpointPolicyKuery: `${_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name: "endpoint"`,
      ensureInCurrentSpace,
      getPolicyNamespace
    };
  }
}
exports.EndpointFleetServicesFactory = EndpointFleetServicesFactory;
/**
 * Checks if data provided (integration policies, agent policies and/or agentIds) are visible in
 * current space
 *
 * @param soClient
 * @param agentService
 * @param agentPolicyService
 * @param packagePolicyService
 * @param integrationPolicyIds
 * @param agentPolicyIds
 * @param agentIds
 *
 * @throws NotFoundError
 */
const checkInCurrentSpace = async ({
  soClient,
  agentService,
  agentPolicyService,
  packagePolicyService,
  integrationPolicyIds = [],
  agentPolicyIds = [],
  agentIds = []
}) => {
  const handlePromiseErrors = err => {
    // We wrap the error with our own Error class so that the API can property return a 404
    if (err instanceof _server.AgentNotFoundError || err instanceof _errors.AgentPolicyNotFoundError || err instanceof _errors.PackagePolicyNotFoundError) {
      throw new _errors2.NotFoundError(err.message, err);
    }
    throw err;
  };
  await Promise.all([agentIds.length ? agentService.getByIds(agentIds).catch(handlePromiseErrors) : null, agentPolicyIds.length ? agentPolicyService.getByIds(soClient, agentPolicyIds).catch(handlePromiseErrors) : null, integrationPolicyIds.length ? packagePolicyService.getByIDs(soClient, integrationPolicyIds).catch(handlePromiseErrors) : null]);
};
const fetchEndpointPolicyNamespace = async ({
  logger,
  soClient,
  packagePolicyService,
  agentPolicyService,
  integrationPolicies
}) => {
  const response = {
    integrationPolicy: {}
  };
  const agentPolicyIdsToRetrieve = new Set();
  const retrievedIntegrationPolicies = {};
  const retrievedAgentPolicies = {};
  if (integrationPolicies.length > 0) {
    var _await$packagePolicyS;
    logger.debug(() => `Retrieving package policies from fleet for:\n${(0, _stringify.stringify)(integrationPolicies)}`);
    const packagePolicies = (_await$packagePolicyS = await packagePolicyService.getByIDs(soClient, integrationPolicies)) !== null && _await$packagePolicyS !== void 0 ? _await$packagePolicyS : [];
    logger.trace(() => `Fleet package policies retrieved:\n${(0, _stringify.stringify)(packagePolicies)}`);
    for (const packagePolicy of packagePolicies) {
      retrievedIntegrationPolicies[packagePolicy.id] = packagePolicy;

      // Integration policy does not have an explicit namespace, which means it
      // inherits it from the associated agent policies, so lets retrieve those
      if (!packagePolicy.namespace) {
        packagePolicy.policy_ids.forEach(agentPolicyId => {
          agentPolicyIdsToRetrieve.add(agentPolicyId);
        });
      }
    }
  }
  if (agentPolicyIdsToRetrieve.size > 0) {
    const ids = Array.from(agentPolicyIdsToRetrieve);
    logger.debug(() => `Retrieving agent policies from fleet for:\n${(0, _stringify.stringify)(ids)}`);
    const agentPolicies = await agentPolicyService.getByIds(soClient, ids);
    logger.trace(() => `Fleet agent policies retrieved:\n${(0, _stringify.stringify)(agentPolicies)}`);
    for (const agentPolicy of agentPolicies) {
      retrievedAgentPolicies[agentPolicy.id] = agentPolicy;
    }
  }
  for (const integrationPolicyId of integrationPolicies) {
    const integrationPolicyNamespace = retrievedIntegrationPolicies[integrationPolicyId].namespace;
    response.integrationPolicy[integrationPolicyId] = integrationPolicyNamespace ? [integrationPolicyNamespace] : retrievedIntegrationPolicies[integrationPolicyId].policy_ids.map(agentPolicyId => {
      return retrievedAgentPolicies[agentPolicyId].namespace;
    });
  }
  logger.debug(() => `Policy namespaces:\n${(0, _stringify.stringify)(response)}`);
  return response;
};