"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.scheduleRuleRun = exports.getRuleIdsWithGaps = exports.getGapsSummaryByRuleIds = exports.findGapsForRule = exports.findBackfillsForRules = exports.fillGapByIdForRule = exports.deleteBackfill = void 0;
var _common = require("@kbn/alerting-plugin/common");
var _datemath = _interopRequireDefault(require("@kbn/datemath"));
var _kibana = require("../../../common/lib/kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Schedule rules run over a specified time range
 *
 * @param ruleIds `rule_id`s of each rule to be backfilled
 * @param timeRange the time range over which the backfill should apply
 *
 * @throws An error if response is not OK
 */
const scheduleRuleRun = async ({
  ruleIds,
  timeRange
}) => {
  const params = ruleIds.map(ruleId => {
    return {
      rule_id: ruleId,
      ranges: [{
        start: timeRange.startDate.toISOString(),
        end: timeRange.endDate.toISOString()
      }]
    };
  });
  return _kibana.KibanaServices.get().http.fetch(_common.INTERNAL_ALERTING_BACKFILL_SCHEDULE_API_PATH, {
    method: 'POST',
    version: '2023-10-31',
    body: JSON.stringify(params)
  });
};

/**
 * Find backfills for the given rule IDs
 * @param ruleIds string[]
 * @param signal? AbortSignal
 * @returns
 */
exports.scheduleRuleRun = scheduleRuleRun;
const findBackfillsForRules = async ({
  ruleIds,
  page,
  perPage,
  signal,
  sortField = 'createdAt',
  sortOrder = 'desc'
}) => _kibana.KibanaServices.get().http.fetch(_common.INTERNAL_ALERTING_BACKFILL_FIND_API_PATH, {
  method: 'POST',
  query: {
    rule_ids: ruleIds.join(','),
    page,
    per_page: perPage,
    sort_field: sortField,
    sort_order: sortOrder
  },
  signal
});

/**
 * Delete backfill by ID
 * @param backfillId
 * @returns
 */
exports.findBackfillsForRules = findBackfillsForRules;
const deleteBackfill = async ({
  backfillId
}) => {
  return _kibana.KibanaServices.get().http.fetch(`${_common.INTERNAL_ALERTING_BACKFILL_API_PATH}/${backfillId}`, {
    method: 'DELETE'
  });
};

/**
 * Find rules with gaps
 * @param ruleIds string[]
 * @param signal? AbortSignal
 * @returns
 */
exports.deleteBackfill = deleteBackfill;
const getRuleIdsWithGaps = async ({
  signal,
  start,
  end,
  statuses = [_common.gapStatus.UNFILLED, _common.gapStatus.PARTIALLY_FILLED],
  hasUnfilledIntervals,
  hasInProgressIntervals,
  hasFilledIntervals
}) => _kibana.KibanaServices.get().http.fetch(_common.INTERNAL_ALERTING_GAPS_GET_RULES_API_PATH, {
  method: 'POST',
  body: JSON.stringify({
    start,
    end,
    statuses,
    ...(hasUnfilledIntervals !== undefined && {
      has_unfilled_intervals: hasUnfilledIntervals
    }),
    ...(hasInProgressIntervals !== undefined && {
      has_in_progress_intervals: hasInProgressIntervals
    }),
    ...(hasFilledIntervals !== undefined && {
      has_filled_intervals: hasFilledIntervals
    })
  }),
  signal
});

/**
 * Find total gap info for the given rule IDs
 * @param ruleIds string[]
 * @param signal? AbortSignal
 * @returns
 */
exports.getRuleIdsWithGaps = getRuleIdsWithGaps;
const getGapsSummaryByRuleIds = async ({
  signal,
  start,
  end,
  ruleIds
}) => _kibana.KibanaServices.get().http.fetch(_common.INTERNAL_ALERTING_GAPS_GET_SUMMARY_BY_RULE_IDS_API_PATH, {
  signal,
  method: 'POST',
  body: JSON.stringify({
    start,
    end,
    rule_ids: ruleIds
  })
});

/**
 * Find gaps for the given rule ID
 * @param ruleIds string[]
 * @param signal? AbortSignal
 * @returns
 */
exports.getGapsSummaryByRuleIds = getGapsSummaryByRuleIds;
const findGapsForRule = async ({
  ruleId,
  page,
  perPage,
  signal,
  sortField = '@timestamp',
  sortOrder = 'desc',
  start,
  end,
  statuses
}) => {
  const startDate = _datemath.default.parse(start);
  const endDate = _datemath.default.parse(end, {
    roundUp: true
  });
  return _kibana.KibanaServices.get().http.fetch(_common.INTERNAL_ALERTING_GAPS_FIND_API_PATH, {
    method: 'POST',
    body: JSON.stringify({
      rule_id: ruleId,
      page,
      per_page: perPage,
      sort_field: sortField,
      sort_order: sortOrder,
      start: startDate === null || startDate === void 0 ? void 0 : startDate.utc().toISOString(),
      end: endDate === null || endDate === void 0 ? void 0 : endDate.utc().toISOString(),
      statuses
    }),
    signal
  });
};

/**
 * Fill gap by Id for the given rule ID
 * @param ruleIds string[]
 * @param signal? AbortSignal
 * @returns
 */
exports.findGapsForRule = findGapsForRule;
const fillGapByIdForRule = async ({
  ruleId,
  gapId,
  signal
}) => _kibana.KibanaServices.get().http.fetch(_common.INTERNAL_ALERTING_GAPS_FILL_BY_ID_API_PATH, {
  method: 'POST',
  query: {
    rule_id: ruleId,
    gap_id: gapId
  },
  signal
});
exports.fillGapByIdForRule = fillGapByIdForRule;