"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.solutionFormatter = exports.registerDeepLinksUpdater = void 0;
var _rxjs = require("rxjs");
var _links = require("@kbn/security-solution-navigation/links");
var _application_links_updater = require("./application_links_updater");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// TODO: remove after rollout https://github.com/elastic/kibana/issues/179572
const classicFormatter = appLinks => appLinks.map(appLink => {
  var _appLink$visibleIn, _appLink$links;
  const visibleIn = new Set((_appLink$visibleIn = appLink.visibleIn) !== null && _appLink$visibleIn !== void 0 ? _appLink$visibleIn : []);
  if (!appLink.globalSearchDisabled) {
    visibleIn.add('globalSearch');
  }
  if (appLink.globalNavPosition != null) {
    visibleIn.add('sideNav');
  }
  const deepLink = {
    id: appLink.id,
    path: appLink.path,
    title: appLink.title,
    visibleIn: Array.from(visibleIn),
    ...(appLink.globalNavPosition != null ? {
      order: appLink.globalNavPosition
    } : {}),
    ...(appLink.globalSearchKeywords != null ? {
      keywords: appLink.globalSearchKeywords
    } : {}),
    ...(appLink.links && (_appLink$links = appLink.links) !== null && _appLink$links !== void 0 && _appLink$links.length ? {
      deepLinks: classicFormatter(appLink.links)
    } : {})
  };
  return deepLink;
});

/**
 * Converts the navigation tree to a deepLinks hierarchy format using the application normalized links.
 * @param navigationTree - The navigation tree to convert
 * @param normalizedLinks - The normalized links to use for formatting
 * @returns The formatted deep links
 */
const solutionFormatter = (navigationTree, normalizedLinks) => {
  const {
    body,
    footer = []
  } = navigationTree;
  const nodes = [];
  [...body, ...footer].forEach(rootNode => {
    if (rootNode.type === 'navGroup') {
      nodes.push(...rootNode.children);
    } else {
      nodes.push(rootNode);
    }
  });
  return solutionNodesFormatter(nodes, normalizedLinks);
};
exports.solutionFormatter = solutionFormatter;
const solutionNodesFormatter = (navigationNodes, normalizedLinks) => {
  const deepLinks = [];
  navigationNodes.forEach(node => {
    // Process links without an id: external links or second level groups
    if (!node.id && node.children) {
      deepLinks.push(...solutionNodesFormatter(node.children, normalizedLinks));
      return;
    }

    // Process top level groups
    const linkGroup = _links.SecurityLinkGroup[node.id];
    if (linkGroup) {
      var _node$children;
      const childrenLinks = solutionNodesFormatter((_node$children = node.children) !== null && _node$children !== void 0 ? _node$children : [], normalizedLinks);
      if (childrenLinks.length > 0) {
        deepLinks.push({
          id: node.id,
          title: linkGroup.title,
          deepLinks: childrenLinks
        });
      }
      return;
    }

    // Process security links
    const appLink = normalizedLinks[node.id];
    if (appLink) {
      const deepLink = formatDeepLink(appLink);
      if (node.children) {
        const childrenLinks = solutionNodesFormatter(node.children, normalizedLinks);
        if (childrenLinks.length > 0) {
          deepLink.deepLinks = childrenLinks;
        }
      }
      deepLinks.push(deepLink);
      return;
    }

    // Process other links with id other than security links or groups
    if (node.children) {
      deepLinks.push(...solutionNodesFormatter(node.children, normalizedLinks));
    }
  });
  return deepLinks;
};
const formatDeepLink = appLink => {
  var _appLink$visibleIn2;
  const visibleIn = new Set((_appLink$visibleIn2 = appLink.visibleIn) !== null && _appLink$visibleIn2 !== void 0 ? _appLink$visibleIn2 : []);
  if (!appLink.globalSearchDisabled) {
    visibleIn.add('globalSearch');
  }
  if (!appLink.sideNavDisabled) {
    visibleIn.add('sideNav');
  }
  const deepLink = {
    id: appLink.id,
    path: appLink.path,
    title: appLink.title,
    visibleIn: Array.from(visibleIn),
    ...(appLink.globalSearchKeywords != null ? {
      keywords: appLink.globalSearchKeywords
    } : {})
  };
  return deepLink;
};

/**
 * Registers any change in appLinks to be updated in app deepLinks
 */
const registerDeepLinksUpdater = (appUpdater$, navigationTree$) => {
  return navigationTree$.pipe((0, _rxjs.combineLatestWith)(_application_links_updater.applicationLinksUpdater.links$, _application_links_updater.applicationLinksUpdater.normalizedLinks$), (0, _rxjs.debounceTime)(100) // Debounce to avoid too many updates
  ).subscribe(([navigationTree, appLinks, normalizedLinks]) => {
    const deepLinks = navigationTree == null ? classicFormatter(appLinks) : solutionFormatter(navigationTree, normalizedLinks);
    appUpdater$.next(() => ({
      deepLinks
    }));
  });
};
exports.registerDeepLinksUpdater = registerDeepLinksUpdater;