"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AnyRule = void 0;
var _i18n = require("@kbn/i18n");
var _except_all_rule = require("./except_all_rule");
var _except_any_rule = require("./except_any_rule");
var _rule_group = require("./rule_group");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Represents a group of rules in which at least one must evaluate to true.
 */
class AnyRule extends _rule_group.RuleGroup {
  constructor(rules = []) {
    super();
    this.rules = rules;
  }

  /** {@see RuleGroup.getRules} */
  getRules() {
    return [...this.rules];
  }

  /** {@see RuleGroup.getDisplayTitle} */
  getDisplayTitle() {
    return _i18n.i18n.translate('xpack.security.management.editRoleMapping.anyRule.displayTitle', {
      defaultMessage: 'Any are true'
    });
  }

  /** {@see RuleGroup.replaceRule} */
  replaceRule(ruleIndex, rule) {
    this.rules.splice(ruleIndex, 1, rule);
  }

  /** {@see RuleGroup.removeRule} */
  removeRule(ruleIndex) {
    this.rules.splice(ruleIndex, 1);
  }

  /** {@see RuleGroup.addRule} */
  addRule(rule) {
    this.rules.push(rule);
  }

  /** {@see RuleGroup.canContainRules} */
  canContainRules(rules) {
    const forbiddenRules = [_except_all_rule.ExceptAllRule, _except_any_rule.ExceptAnyRule];
    return rules.every(candidate => !forbiddenRules.some(forbidden => candidate instanceof forbidden));
  }

  /** {@see RuleGroup.clone} */
  clone() {
    return new AnyRule(this.rules.map(r => r.clone()));
  }

  /** {@see RuleGroup.toRaw} */
  toRaw() {
    return {
      any: [...this.rules.map(rule => rule.toRaw())]
    };
  }
}
exports.AnyRule = AnyRule;