"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calendars = calendars;
var _app = require("../../common/constants/app");
var _error_wrapper = require("../client/error_wrapper");
var _calendars_schema = require("./schemas/calendars_schema");
var _calendar = require("../models/calendar");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getAllCalendars(mlClient) {
  const cal = new _calendar.CalendarManager(mlClient);
  return cal.getAllCalendars();
}
function getCalendar(mlClient, calendarId) {
  const cal = new _calendar.CalendarManager(mlClient);
  return cal.getCalendar(calendarId);
}
function newCalendar(mlClient, calendar) {
  const cal = new _calendar.CalendarManager(mlClient);
  return cal.newCalendar(calendar);
}
function updateCalendar(mlClient, calendarId, calendar) {
  const cal = new _calendar.CalendarManager(mlClient);
  return cal.updateCalendar(calendarId, calendar);
}
function deleteCalendar(mlClient, calendarId) {
  const cal = new _calendar.CalendarManager(mlClient);
  return cal.deleteCalendar(calendarId);
}
function getCalendarsByIds(mlClient, calendarIds) {
  const cal = new _calendar.CalendarManager(mlClient);
  return cal.getCalendarsByIds(calendarIds);
}
function calendars({
  router,
  routeGuard
}) {
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/calendars`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetCalendars']
      }
    },
    summary: 'Gets calendars',
    description: 'Gets calendars - size limit has been explicitly set to 10000'
  }).addVersion({
    version: '1',
    validate: false
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    response
  }) => {
    try {
      const resp = await getAllCalendars(mlClient);
      return response.ok({
        body: resp
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/calendars/{calendarIds}`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetCalendars']
      }
    },
    summary: 'Gets a calendar',
    description: 'Gets a calendar by id'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _calendars_schema.calendarIdsSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    let returnValue;
    try {
      const calendarIds = request.params.calendarIds.split(',');
      if (calendarIds.length === 1) {
        returnValue = await getCalendar(mlClient, calendarIds[0]);
      } else {
        returnValue = await getCalendarsByIds(mlClient, calendarIds);
      }
      return response.ok({
        body: returnValue
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.put({
    path: `${_app.ML_INTERNAL_BASE_PATH}/calendars`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canCreateCalendar']
      }
    },
    summary: 'Creates a calendar',
    description: 'Creates a calendar'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: _calendars_schema.calendarSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const body = request.body;
      // @ts-expect-error event interface incorrect
      const resp = await newCalendar(mlClient, body);
      return response.ok({
        body: resp
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.put({
    path: `${_app.ML_INTERNAL_BASE_PATH}/calendars/{calendarId}`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canCreateCalendar']
      }
    },
    summary: 'Updates a calendar',
    description: 'Updates a calendar'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _calendars_schema.calendarIdSchema,
        body: _calendars_schema.calendarSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const {
        calendarId
      } = request.params;
      const body = request.body;
      // @ts-expect-error event interface incorrect
      const resp = await updateCalendar(mlClient, calendarId, body);
      return response.ok({
        body: resp
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.delete({
    path: `${_app.ML_INTERNAL_BASE_PATH}/calendars/{calendarId}`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canDeleteCalendar']
      }
    },
    summary: 'Deletes a calendar',
    description: 'Deletes a calendar'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _calendars_schema.calendarIdSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const {
        calendarId
      } = request.params;
      const resp = await deleteCalendar(mlClient, calendarId);
      return response.ok({
        body: resp
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
}