"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.resultsServiceRxProvider = resultsServiceRxProvider;
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _mlIsPopulatedObject = require("@kbn/ml-is-populated-object");
var _mlAnomalyUtils = require("@kbn/ml-anomaly-utils");
var _mlRuntimeFieldUtils = require("@kbn/ml-runtime-field-utils");
var _job_utils = require("../../../../common/util/job_utils");
var _validation_utils = require("../../../../common/util/validation_utils");
var _datafeed_utils = require("../../../../common/util/datafeed_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Queries Elasticsearch to obtain metric aggregation results.
// index can be a String, or String[], of index names to search.
// entityFields parameter must be an array, with each object in the array having 'fieldName'
// and 'fieldValue' properties.
// Extra query object can be supplied, or pass null if no additional query
// to that built from the supplied entity fields.
// Returned response contains a results property containing the requested aggregation.

function resultsServiceRxProvider(mlApi) {
  return {
    getMetricData(index, entityFields, query, metricFunction,
    // ES aggregation name
    metricFieldName, summaryCountFieldName, timeFieldName, earliestMs, latestMs, intervalMs, datafeedConfig) {
      const scriptFields = datafeedConfig === null || datafeedConfig === void 0 ? void 0 : datafeedConfig.script_fields;
      const aggFields = (0, _datafeed_utils.getDatafeedAggregations)(datafeedConfig);

      // Build the criteria to use in the bool filter part of the request.
      // Add criteria for the time range, entity fields,
      // plus any additional supplied query.
      const shouldCriteria = [];
      const mustCriteria = [{
        range: {
          [timeFieldName]: {
            gte: earliestMs,
            lte: latestMs,
            format: 'epoch_millis'
          }
        }
      }, ...(query ? [query] : [])];
      entityFields.forEach(entity => {
        if (entity.fieldValue.length !== 0) {
          mustCriteria.push({
            term: {
              [entity.fieldName]: entity.fieldValue
            }
          });
        } else {
          // Add special handling for blank entity field values, checking for either
          // an empty string or the field not existing.
          shouldCriteria.push({
            bool: {
              must: [{
                term: {
                  [entity.fieldName]: ''
                }
              }]
            }
          });
          shouldCriteria.push({
            bool: {
              must_not: [{
                exists: {
                  field: entity.fieldName
                }
              }]
            }
          });
        }
      });
      const body = {
        query: {
          bool: {
            must: mustCriteria
          }
        },
        size: 0,
        _source: false,
        aggs: {
          byTime: {
            date_histogram: {
              field: timeFieldName,
              fixed_interval: `${intervalMs}ms`,
              min_doc_count: 0
            }
          }
        },
        ...((0, _mlRuntimeFieldUtils.isRuntimeMappings)(datafeedConfig === null || datafeedConfig === void 0 ? void 0 : datafeedConfig.runtime_mappings) ? {
          runtime_mappings: datafeedConfig === null || datafeedConfig === void 0 ? void 0 : datafeedConfig.runtime_mappings
        } : {})
      };
      if (shouldCriteria.length > 0) {
        body.query.bool.should = shouldCriteria;
        body.query.bool.minimum_should_match = shouldCriteria.length / 2;
      }
      body.aggs.byTime.aggs = {};
      if (metricFieldName !== undefined && metricFieldName !== '' && metricFunction) {
        const metricAgg = {
          [metricFunction]: {}
        };
        if (scriptFields !== undefined && scriptFields[metricFieldName] !== undefined) {
          metricAgg[metricFunction].script = scriptFields[metricFieldName].script;
        } else {
          metricAgg[metricFunction].field = metricFieldName;
        }
        if (metricFunction === 'percentiles') {
          metricAgg[metricFunction].percents = [_job_utils.ML_MEDIAN_PERCENTS];
        }

        // when the field is an aggregation field, because the field doesn't actually exist in the indices
        // we need to pass all the sub aggs from the original datafeed config
        // so that we can access the aggregated field
        if ((0, _mlIsPopulatedObject.isPopulatedObject)(aggFields)) {
          var _aggFields$accessor$a;
          // first item under aggregations can be any name, not necessarily 'buckets'
          const accessor = Object.keys(aggFields)[0];
          const tempAggs = {
            ...((_aggFields$accessor$a = aggFields[accessor].aggs) !== null && _aggFields$accessor$a !== void 0 ? _aggFields$accessor$a : aggFields[accessor].aggregations)
          };
          const foundValue = (0, _validation_utils.findAggField)(tempAggs, metricFieldName);
          if (foundValue !== undefined) {
            tempAggs.metric = foundValue;
            delete tempAggs[metricFieldName];
          }
          body.aggs.byTime.aggs = tempAggs;
        } else {
          body.aggs.byTime.aggs.metric = metricAgg;
        }
      } else {
        // if metricFieldName is not defined, it's probably a variation of the non zero count function
        // refer to buildConfigFromDetector
        if (summaryCountFieldName !== undefined && metricFunction === _mlAnomalyUtils.ES_AGGREGATION.CARDINALITY) {
          // if so, check if summaryCountFieldName is an aggregation field
          if (typeof aggFields === 'object' && Object.keys(aggFields).length > 0) {
            var _aggFields$accessor$a2;
            // first item under aggregations can be any name, not necessarily 'buckets'
            const accessor = Object.keys(aggFields)[0];
            const tempAggs = {
              ...((_aggFields$accessor$a2 = aggFields[accessor].aggs) !== null && _aggFields$accessor$a2 !== void 0 ? _aggFields$accessor$a2 : aggFields[accessor].aggregations)
            };
            const foundCardinalityField = (0, _validation_utils.findAggField)(tempAggs, summaryCountFieldName);
            if (foundCardinalityField !== undefined) {
              tempAggs.metric = foundCardinalityField;
            }
            body.aggs.byTime.aggs = tempAggs;
          }
        }
      }
      return mlApi.esSearch$({
        index,
        body
      }).pipe((0, _rxjs.map)(resp => {
        var _resp$aggregations$by, _resp$aggregations, _resp$aggregations$by2;
        const obj = {
          success: true,
          results: {}
        };
        const dataByTime = (_resp$aggregations$by = resp === null || resp === void 0 ? void 0 : (_resp$aggregations = resp.aggregations) === null || _resp$aggregations === void 0 ? void 0 : (_resp$aggregations$by2 = _resp$aggregations.byTime) === null || _resp$aggregations$by2 === void 0 ? void 0 : _resp$aggregations$by2.buckets) !== null && _resp$aggregations$by !== void 0 ? _resp$aggregations$by : [];
        dataByTime.forEach(dataForTime => {
          if (metricFunction === 'count') {
            obj.results[dataForTime.key] = dataForTime.doc_count;
          } else {
            var _dataForTime$metric, _dataForTime$metric2;
            const value = dataForTime === null || dataForTime === void 0 ? void 0 : (_dataForTime$metric = dataForTime.metric) === null || _dataForTime$metric === void 0 ? void 0 : _dataForTime$metric.value;
            const values = dataForTime === null || dataForTime === void 0 ? void 0 : (_dataForTime$metric2 = dataForTime.metric) === null || _dataForTime$metric2 === void 0 ? void 0 : _dataForTime$metric2.values;
            if (dataForTime.doc_count === 0) {
              obj.results[dataForTime.key] = null;
            } else if (value !== undefined) {
              obj.results[dataForTime.key] = value;
            } else if (values !== undefined) {
              // Percentiles agg currently returns NaN rather than null when none of the docs in the
              // bucket contain the field used in the aggregation
              // (see elasticsearch issue https://github.com/elastic/elasticsearch/issues/29066).
              // Store as null, so values can be handled in the same manner downstream as other aggs
              // (min, mean, max) which return null.
              const medianValues = values[_job_utils.ML_MEDIAN_PERCENTS];
              obj.results[dataForTime.key] = !isNaN(medianValues) ? medianValues : null;
            } else {
              obj.results[dataForTime.key] = null;
            }
          }
        });
        return obj;
      }));
    },
    getModelPlotOutput(jobId, detectorIndex, criteriaFields, earliestMs, latestMs, intervalMs, aggType) {
      const obj = {
        success: true,
        results: {}
      };

      // if an aggType object has been passed in, use it.
      // otherwise default to min and max aggs for the upper and lower bounds
      const modelAggs = aggType === undefined ? {
        max: 'max',
        min: 'min'
      } : {
        max: aggType.max,
        min: aggType.min
      };

      // Build the criteria to use in the bool filter part of the request.
      // Add criteria for the job ID and time range.
      const mustCriteria = [{
        term: {
          job_id: jobId
        }
      }, {
        range: {
          timestamp: {
            gte: earliestMs,
            lte: latestMs,
            format: 'epoch_millis'
          }
        }
      }];

      // Add in term queries for each of the specified criteria.
      (0, _lodash.each)(criteriaFields, criteria => {
        mustCriteria.push({
          term: {
            [criteria.fieldName]: criteria.fieldValue
          }
        });
      });

      // Add criteria for the detector index. Results from jobs created before 6.1 will not
      // contain a detector_index field, so use a should criteria with a 'not exists' check.
      const shouldCriteria = [{
        term: {
          detector_index: detectorIndex
        }
      }, {
        bool: {
          must_not: [{
            exists: {
              field: 'detector_index'
            }
          }]
        }
      }];
      return mlApi.results.anomalySearch$({
        body: {
          size: 0,
          query: {
            bool: {
              filter: [{
                query_string: {
                  query: 'result_type:model_plot',
                  analyze_wildcard: true
                }
              }, {
                bool: {
                  must: mustCriteria,
                  should: shouldCriteria,
                  minimum_should_match: 1
                }
              }]
            }
          },
          aggs: {
            times: {
              date_histogram: {
                field: 'timestamp',
                fixed_interval: `${intervalMs}ms`,
                min_doc_count: 0
              },
              aggs: {
                actual: {
                  avg: {
                    field: 'actual'
                  }
                },
                modelUpper: {
                  [modelAggs.max]: {
                    field: 'model_upper'
                  }
                },
                modelLower: {
                  [modelAggs.min]: {
                    field: 'model_lower'
                  }
                }
              }
            }
          }
        }
      }, [jobId]).pipe((0, _rxjs.map)(resp => {
        const aggregationsByTime = (0, _lodash.get)(resp, ['aggregations', 'times', 'buckets'], []);
        (0, _lodash.each)(aggregationsByTime, dataForTime => {
          const time = dataForTime.key;
          const modelUpper = (0, _lodash.get)(dataForTime, ['modelUpper', 'value']);
          const modelLower = (0, _lodash.get)(dataForTime, ['modelLower', 'value']);
          const actual = (0, _lodash.get)(dataForTime, ['actual', 'value']);
          obj.results[time] = {
            actual,
            modelUpper: modelUpper === undefined || isFinite(modelUpper) === false ? null : modelUpper,
            modelLower: modelLower === undefined || isFinite(modelLower) === false ? null : modelLower
          };
        });
        return obj;
      }));
    },
    // Queries Elasticsearch to obtain the record level results matching the given criteria,
    // for the specified job(s), time range, and record score threshold.
    // criteriaFields parameter must be an array, with each object in the array having 'fieldName'
    // 'fieldValue' properties.
    // Pass an empty array or ['*'] to search over all job IDs.
    getRecordsForCriteria(jobIds, criteriaFields, threshold, earliestMs, latestMs, maxResults, functionDescription) {
      const obj = {
        success: true,
        records: []
      };

      // Build the criteria to use in the bool filter part of the request.
      // Add criteria for the time range, record score, plus any specified job IDs.
      const boolCriteria = [{
        range: {
          timestamp: {
            gte: earliestMs,
            lte: latestMs,
            format: 'epoch_millis'
          }
        }
      }, {
        range: {
          record_score: {
            gte: threshold
          }
        }
      }];
      if (jobIds && jobIds.length > 0 && !(jobIds.length === 1 && jobIds[0] === '*')) {
        let jobIdFilterStr = '';
        (0, _lodash.each)(jobIds, (jobId, i) => {
          if (i > 0) {
            jobIdFilterStr += ' OR ';
          }
          jobIdFilterStr += 'job_id:';
          jobIdFilterStr += jobId;
        });
        boolCriteria.push({
          query_string: {
            analyze_wildcard: false,
            query: jobIdFilterStr
          }
        });
      }

      // Add in term queries for each of the specified criteria.
      (0, _lodash.each)(criteriaFields, criteria => {
        boolCriteria.push({
          term: {
            [criteria.fieldName]: criteria.fieldValue
          }
        });
      });
      if (functionDescription !== undefined) {
        const mlFunctionToPlotIfMetric = functionDescription !== undefined ? _mlAnomalyUtils.aggregationTypeTransform.toML(functionDescription) : functionDescription;
        boolCriteria.push({
          term: {
            function_description: mlFunctionToPlotIfMetric
          }
        });
      }
      return mlApi.results.anomalySearch$({
        body: {
          size: maxResults !== undefined ? maxResults : 100,
          query: {
            bool: {
              filter: [{
                query_string: {
                  query: 'result_type:record',
                  analyze_wildcard: false
                }
              }, {
                bool: {
                  must: boolCriteria
                }
              }]
            }
          },
          sort: [{
            record_score: {
              order: 'desc'
            }
          }]
        }
      }, jobIds).pipe((0, _rxjs.map)(resp => {
        if (resp.hits.total.value > 0) {
          (0, _lodash.each)(resp.hits.hits, hit => {
            obj.records.push(hit._source);
          });
        }
        return obj;
      }));
    },
    // Obtains a list of scheduled events by job ID and time.
    // Pass an empty array or ['*'] to search over all job IDs.
    // Returned response contains a events property, which will only
    // contains keys for jobs which have scheduled events for the specified time range.
    getScheduledEventsByBucket(jobIds, earliestMs, latestMs, intervalMs, maxJobs, maxEvents) {
      const obj = {
        success: true,
        events: {}
      };

      // Build the criteria to use in the bool filter part of the request.
      // Adds criteria for the time range plus any specified job IDs.
      const boolCriteria = [{
        range: {
          timestamp: {
            gte: earliestMs,
            lte: latestMs,
            format: 'epoch_millis'
          }
        }
      }, {
        exists: {
          field: 'scheduled_events'
        }
      }];
      if (jobIds && jobIds.length > 0 && !(jobIds.length === 1 && jobIds[0] === '*')) {
        let jobIdFilterStr = '';
        (0, _lodash.each)(jobIds, (jobId, i) => {
          jobIdFilterStr += `${i > 0 ? ' OR ' : ''}job_id:${jobId}`;
        });
        boolCriteria.push({
          query_string: {
            analyze_wildcard: false,
            query: jobIdFilterStr
          }
        });
      }
      return mlApi.results.anomalySearch$({
        body: {
          size: 0,
          query: {
            bool: {
              filter: [{
                query_string: {
                  query: 'result_type:bucket',
                  analyze_wildcard: false
                }
              }, {
                bool: {
                  must: boolCriteria
                }
              }]
            }
          },
          aggs: {
            jobs: {
              terms: {
                field: 'job_id',
                min_doc_count: 1,
                size: maxJobs
              },
              aggs: {
                times: {
                  date_histogram: {
                    field: 'timestamp',
                    fixed_interval: `${intervalMs}ms`,
                    min_doc_count: 1
                  },
                  aggs: {
                    events: {
                      terms: {
                        field: 'scheduled_events',
                        size: maxEvents
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }, jobIds).pipe((0, _rxjs.map)(resp => {
        const dataByJobId = (0, _lodash.get)(resp, ['aggregations', 'jobs', 'buckets'], []);
        (0, _lodash.each)(dataByJobId, dataForJob => {
          const jobId = dataForJob.key;
          const resultsForTime = {};
          const dataByTime = (0, _lodash.get)(dataForJob, ['times', 'buckets'], []);
          (0, _lodash.each)(dataByTime, dataForTime => {
            const time = dataForTime.key;
            const events = (0, _lodash.get)(dataForTime, ['events', 'buckets']);
            resultsForTime[time] = events.map(e => e.key);
          });
          obj.events[jobId] = resultsForTime;
        });
        return obj;
      }));
    },
    fetchPartitionFieldsValues(jobId, searchTerm, criteriaFields, earliestMs, latestMs) {
      return mlApi.results.fetchPartitionFieldsValues(jobId, searchTerm, criteriaFields, earliestMs, latestMs);
    },
    // Queries Elasticsearch to obtain the record level results containing the specified influencer(s),
    // for the specified job(s), time range, and record score threshold.
    // influencers parameter must be an array, with each object in the array having 'fieldName'
    // 'fieldValue' properties. The influencer array uses 'should' for the nested bool query,
    // so this returns record level results which have at least one of the influencers.
    // Pass an empty array or ['*'] to search over all job IDs.
    getRecordsForInfluencer$(jobIds, influencers, threshold, earliestMs, latestMs, maxResults, influencersFilterQuery) {
      const obj = {
        success: true,
        records: []
      };

      // Build the criteria to use in the bool filter part of the request.
      // Add criteria for the time range, record score, plus any specified job IDs.
      const boolCriteria = [{
        range: {
          timestamp: {
            gte: earliestMs,
            lte: latestMs,
            format: 'epoch_millis'
          }
        }
      }, {
        range: {
          record_score: {
            gte: threshold
          }
        }
      }];
      if (jobIds && jobIds.length > 0 && !(jobIds.length === 1 && jobIds[0] === '*')) {
        let jobIdFilterStr = '';
        (0, _lodash.each)(jobIds, (jobId, i) => {
          if (i > 0) {
            jobIdFilterStr += ' OR ';
          }
          jobIdFilterStr += 'job_id:';
          jobIdFilterStr += jobId;
        });
        boolCriteria.push({
          query_string: {
            analyze_wildcard: false,
            query: jobIdFilterStr
          }
        });
      }
      if (influencersFilterQuery !== undefined) {
        boolCriteria.push(influencersFilterQuery);
      }

      // Add a nested query to filter for each of the specified influencers.
      if (influencers.length > 0) {
        boolCriteria.push({
          bool: {
            should: influencers.map(influencer => {
              return {
                nested: {
                  path: 'influencers',
                  query: {
                    bool: {
                      must: [{
                        match: {
                          'influencers.influencer_field_name': influencer.fieldName
                        }
                      }, {
                        match: {
                          'influencers.influencer_field_values': influencer.fieldValue
                        }
                      }]
                    }
                  }
                }
              };
            }),
            minimum_should_match: 1
          }
        });
      }
      return mlApi.results.anomalySearch$({
        body: {
          size: maxResults !== undefined ? maxResults : 100,
          query: {
            bool: {
              filter: [{
                query_string: {
                  query: 'result_type:record',
                  analyze_wildcard: false
                }
              }, {
                bool: {
                  must: boolCriteria
                }
              }]
            }
          },
          sort: [{
            record_score: {
              order: 'desc'
            }
          }]
        }
      }, jobIds).pipe((0, _rxjs.map)(resp => {
        if (resp.hits.total.value > 0) {
          (0, _lodash.each)(resp.hits.hits, hit => {
            obj.records.push(hit._source);
          });
        }
        return obj;
      }));
    }
  };
}