"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseExperimentalConfigValue = exports.isValidExperimentalValue = exports.getExperimentalAllowedValues = exports.allowedExperimentalValues = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const _allowedExperimentalValues = {
  createPackagePolicyMultiPageLayout: true,
  packageVerification: true,
  diagnosticFileUploadEnabled: true,
  displayAgentMetrics: true,
  showIntegrationsSubcategories: true,
  agentFqdnMode: true,
  showExperimentalShipperOptions: false,
  agentTamperProtectionEnabled: true,
  secretsStorage: true,
  kafkaOutput: true,
  outputSecretsStorage: true,
  remoteESOutput: true,
  enableStrictKQLValidation: true,
  subfeaturePrivileges: true,
  advancedPolicySettings: true,
  useSpaceAwareness: false,
  enableReusableIntegrationPolicies: true,
  asyncDeployPolicies: true,
  enableExportCSV: true,
  enabledUpgradeAgentlessDeploymentsTask: true,
  enableAutomaticAgentUpgrades: true
};

/**
 * A list of allowed values that can be used in `xpack.fleet.enableExperimental`.
 * This object is then used to validate and parse the value entered.
 */
const allowedExperimentalValues = exports.allowedExperimentalValues = Object.freeze({
  ..._allowedExperimentalValues
});
const allowedKeys = Object.keys(allowedExperimentalValues);

/**
 * Parses the string value used in `xpack.fleet.enableExperimental` kibana configuration,
 * which should be a string of values delimited by a comma (`,`)
 *
 * @param configValue
 */
const parseExperimentalConfigValue = configValue => {
  const enabledFeatures = {
    ...allowedExperimentalValues
  };
  for (const value of configValue) {
    if (isValidExperimentalValue(value)) {
      enabledFeatures[value] = true;
    }
  }
  return {
    ...allowedExperimentalValues,
    ...enabledFeatures
  };
};
exports.parseExperimentalConfigValue = parseExperimentalConfigValue;
const isValidExperimentalValue = value => {
  return allowedKeys.includes(value);
};
exports.isValidExperimentalValue = isValidExperimentalValue;
const getExperimentalAllowedValues = () => [...allowedKeys];
exports.getExperimentalAllowedValues = getExperimentalAllowedValues;