"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.resetOriginalid = exports.isFieldValue = exports.getIndex = exports.flattenObjectPreservingValues = exports.convertResults = exports.convertResultToFieldsAndIndex = exports.addTypeToResults = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const resetOriginalid = result => {
  const {
    _meta: {
      rawHit: {
        __id: id
      }
    }
  } = result;
  return {
    ...result,
    id: {
      raw: id
    }
  };
};
exports.resetOriginalid = resetOriginalid;
const getIndex = result => {
  const {
    _meta: {
      rawHit: {
        _index: index
      }
    }
  } = result;
  return index;
};
exports.getIndex = getIndex;
const isFieldValue = value => {
  if (value === null || typeof value !== 'object') return false;
  return Object.hasOwn(value, 'raw') || Object.hasOwn(value, 'snippet');
};
exports.isFieldValue = isFieldValue;
const flattenObjectPreservingValues = (obj, prefix = '') => {
  if (isFieldValue(obj) && typeof obj.raw !== 'object') return {
    [prefix]: obj
  };
  return Object.keys(obj).reduce((acc, key) => {
    const dot = prefix.length ? prefix + '.' : '';
    const val = obj[key];
    if (typeof val === 'object' && val !== null) {
      Object.assign(acc, flattenObjectPreservingValues(val, key === 'raw' ? prefix : dot + key));
    } else {
      acc[dot + key] = {
        raw: val
      };
    }
    return acc;
  }, {});
};
exports.flattenObjectPreservingValues = flattenObjectPreservingValues;
const convertResults = result => {
  const flattenedResult = flattenObjectPreservingValues(result);
  const unsortedFields = Object.entries(flattenedResult).map(([field, value]) => ({
    field,
    value
  }));
  const sortedFields = unsortedFields.sort((a, b) => a.field.localeCompare(b.field));
  return sortedFields;
};
exports.convertResults = convertResults;
const convertResultToFieldsAndIndex = result => {
  const index = getIndex(result);
  const {
    _meta: _,
    ...withOriginalId
  } = resetOriginalid(result);
  const fields = convertResults(withOriginalId);
  return {
    fields,
    index
  };
};
exports.convertResultToFieldsAndIndex = convertResultToFieldsAndIndex;
const addTypeToResults = (results, fieldTypes) => {
  return results.map(result => {
    const type = fieldTypes[result.field];
    return {
      ...result,
      type
    };
  });
};
exports.addTypeToResults = addTypeToResults;