"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.agentlessAgentService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _https = _interopRequireDefault(require("https"));
var _serverHttpTools = require("@kbn/server-http-tools");
var _axios = _interopRequireDefault(require("axios"));
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _constants = require("../../constants");
var _errors = require("../../errors");
var _app_context = require("../app_context");
var _api_keys = require("../api_keys");
var _fleet_server_host = require("../fleet_server_host");
var _agentless = require("../utils/agentless");
var _agentless2 = require("../../../common/constants/agentless");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class AgentlessAgentService {
  constructor() {
    (0, _defineProperty2.default)(this, "convertCauseErrorsToString", error => {
      if (error.cause instanceof AggregateError) {
        return error.cause.errors.map(e => e.message);
      }
      return error.cause;
    });
    (0, _defineProperty2.default)(this, "handleErrorsWithRetries", async (error, requestConfig, action, logger, retries, id, requestConfigDebugStatus, errorMetadata, traceId) => {
      const hasRetryableStatusError = this.hasRetryableStatusError(error, _agentless2.RETRYABLE_HTTP_STATUSES);
      const hasRetryableCodeError = this.hasRetryableCodeError(error, _agentless2.RETRYABLE_SERVER_CODES);
      if (hasRetryableStatusError || hasRetryableCodeError) {
        await this.retry(async () => await (0, _axios.default)(requestConfig), action, requestConfigDebugStatus, logger, retries, () => this.catchAgentlessApiError(action, error, logger, id, requestConfig, requestConfigDebugStatus, errorMetadata, traceId));
      } else {
        this.catchAgentlessApiError(action, error, logger, id, requestConfig, requestConfigDebugStatus, errorMetadata, traceId);
      }
    });
    (0, _defineProperty2.default)(this, "retry", async (fn, action, requestConfigDebugStatus, logger, retries = _agentless2.MAXIMUM_RETRIES, throwAgentlessError) => {
      for (let i = 0; i < retries; i++) {
        try {
          await fn();
        } catch (e) {
          logger.info(`[Agentless API] Attempt ${i + 1} failed to ${action} agentless deployment, retrying...`);
          if (i === retries - 1) {
            logger.error(`[Agentless API] Reached maximum ${retries} attempts. Failed to ${action} agentless deployment with [REQUEST]: ${requestConfigDebugStatus}`);
            throwAgentlessError();
          }
        }
      }
    });
    (0, _defineProperty2.default)(this, "hasRetryableStatusError", (error, retryableStatusErrors) => {
      var _error$response;
      const status = error === null || error === void 0 ? void 0 : (_error$response = error.response) === null || _error$response === void 0 ? void 0 : _error$response.status;
      return !!status && retryableStatusErrors.some(errorStatus => errorStatus === status);
    });
    (0, _defineProperty2.default)(this, "hasRetryableCodeError", (error, retryableCodeErrors) => {
      const code = error === null || error === void 0 ? void 0 : error.code;
      return !!code && retryableCodeErrors.includes(code);
    });
  }
  async createAgentlessAgent(esClient, soClient, agentlessAgentPolicy) {
    var _apm$currentTransacti, _appContextService$ge, _agentlessConfig$api, _agentlessConfig$api$, _agentlessConfig$api2, _agentlessConfig$api3, _agentlessConfig$api4, _agentlessConfig$api5, _agentlessAgentPolicy;
    const traceId = (_apm$currentTransacti = _elasticApmNode.default.currentTransaction) === null || _apm$currentTransacti === void 0 ? void 0 : _apm$currentTransacti.traceparent;
    const errorMetadata = {
      trace: {
        id: traceId
      }
    };
    const logger = _app_context.appContextService.getLogger();
    logger.debug(`[Agentless API] Creating agentless agent ${agentlessAgentPolicy.id}`);
    const agentlessConfig = (_appContextService$ge = _app_context.appContextService.getConfig()) === null || _appContextService$ge === void 0 ? void 0 : _appContextService$ge.agentless;
    if (!agentlessConfig) {
      logger.error('[Agentless API] Missing agentless configuration', errorMetadata);
      throw new _errors.AgentlessAgentConfigError('missing Agentless API configuration in Kibana');
    }
    if (!(0, _agentless.isAgentlessEnabled)()) {
      logger.error('[Agentless API] Agentless agents are only supported in cloud deployment and serverless projects');
      throw new _errors.AgentlessAgentConfigError('Agentless agents are only supported in cloud deployment and serverless projects');
    }
    if (!agentlessAgentPolicy.supports_agentless) {
      logger.error('[Agentless API] Agentless agent policy does not have agentless enabled');
      throw new _errors.AgentlessAgentConfigError('Agentless agent policy does not have supports_agentless enabled');
    }
    const policyId = agentlessAgentPolicy.id;
    const {
      fleetUrl,
      fleetToken
    } = await this.getFleetUrlAndTokenForAgentlessAgent(esClient, policyId, soClient);
    logger.debug(`[Agentless API] Creating agentless agent with fleetUrl ${fleetUrl} and fleet_token: [REDACTED]`);
    logger.debug(`[Agentless API] Creating agentless agent with TLS cert: ${agentlessConfig !== null && agentlessConfig !== void 0 && (_agentlessConfig$api = agentlessConfig.api) !== null && _agentlessConfig$api !== void 0 && (_agentlessConfig$api$ = _agentlessConfig$api.tls) !== null && _agentlessConfig$api$ !== void 0 && _agentlessConfig$api$.certificate ? '[REDACTED]' : 'undefined'} and TLS key: ${agentlessConfig !== null && agentlessConfig !== void 0 && (_agentlessConfig$api2 = agentlessConfig.api) !== null && _agentlessConfig$api2 !== void 0 && (_agentlessConfig$api3 = _agentlessConfig$api2.tls) !== null && _agentlessConfig$api3 !== void 0 && _agentlessConfig$api3.key ? '[REDACTED]' : 'undefined'}
      and TLS ca: ${agentlessConfig !== null && agentlessConfig !== void 0 && (_agentlessConfig$api4 = agentlessConfig.api) !== null && _agentlessConfig$api4 !== void 0 && (_agentlessConfig$api5 = _agentlessConfig$api4.tls) !== null && _agentlessConfig$api5 !== void 0 && _agentlessConfig$api5.ca ? '[REDACTED]' : 'undefined'}`);
    const tlsConfig = this.createTlsConfig(agentlessConfig);
    const labels = this.getAgentlessTags(agentlessAgentPolicy);
    const requestConfig = {
      url: (0, _agentless.prependAgentlessApiBasePathToEndpoint)(agentlessConfig, '/deployments'),
      data: {
        policy_id: policyId,
        fleet_url: fleetUrl,
        fleet_token: fleetToken,
        resources: (_agentlessAgentPolicy = agentlessAgentPolicy.agentless) === null || _agentlessAgentPolicy === void 0 ? void 0 : _agentlessAgentPolicy.resources,
        labels
      },
      method: 'POST',
      ...this.getHeaders(tlsConfig, traceId)
    };
    const cloudSetup = _app_context.appContextService.getCloud();
    if (!(cloudSetup !== null && cloudSetup !== void 0 && cloudSetup.isServerlessEnabled)) {
      requestConfig.data.stack_version = _app_context.appContextService.getKibanaVersion();
    }
    const requestConfigDebugStatus = this.createRequestConfigDebug(requestConfig);
    logger.debug(`[Agentless API] Creating agentless agent with request config ${requestConfigDebugStatus}`);
    const response = await (0, _axios.default)(requestConfig).catch(error => {
      this.catchAgentlessApiError('create', error, logger, agentlessAgentPolicy.id, requestConfig, requestConfigDebugStatus, errorMetadata, traceId);
    });
    logger.debug(`[Agentless API] Created an agentless agent ${response}`);
    return response;
  }
  async deleteAgentlessAgent(agentlessPolicyId) {
    var _apm$currentTransacti2, _appContextService$ge2;
    const logger = _app_context.appContextService.getLogger();
    const traceId = (_apm$currentTransacti2 = _elasticApmNode.default.currentTransaction) === null || _apm$currentTransacti2 === void 0 ? void 0 : _apm$currentTransacti2.traceparent;
    const agentlessConfig = (_appContextService$ge2 = _app_context.appContextService.getConfig()) === null || _appContextService$ge2 === void 0 ? void 0 : _appContextService$ge2.agentless;
    const tlsConfig = this.createTlsConfig(agentlessConfig);
    const requestConfig = {
      url: (0, _agentless.prependAgentlessApiBasePathToEndpoint)(agentlessConfig, `/deployments/${agentlessPolicyId}`),
      method: 'DELETE',
      ...this.getHeaders(tlsConfig, traceId)
    };
    const errorMetadata = {
      trace: {
        id: traceId
      }
    };
    const requestConfigDebugStatus = this.createRequestConfigDebug(requestConfig);
    logger.debug(`[Agentless API] Start deleting agentless agent for agent policy ${requestConfigDebugStatus}`);
    if (!_agentless.isAgentlessEnabled) {
      logger.error('[Agentless API] Agentless API is not supported. Deleting agentless agent is not supported in non-cloud or non-serverless environments');
    }
    if (!agentlessConfig) {
      logger.error('[Agentless API] kibana.yml is currently missing Agentless API configuration');
    }
    logger.debug(`[Agentless API] Deleting agentless agent with TLS config with certificate`);
    logger.debug(`[Agentless API] Deleting agentless deployment with request config ${requestConfigDebugStatus}`);
    const response = await (0, _axios.default)(requestConfig).catch(error => {
      this.catchAgentlessApiError('delete', error, logger, agentlessPolicyId, requestConfig, requestConfigDebugStatus, errorMetadata, traceId);
    });
    return response;
  }
  async upgradeAgentlessDeployment(policyId) {
    var _apm$currentTransacti3, _appContextService$ge3;
    const logger = _app_context.appContextService.getLogger();
    const traceId = (_apm$currentTransacti3 = _elasticApmNode.default.currentTransaction) === null || _apm$currentTransacti3 === void 0 ? void 0 : _apm$currentTransacti3.traceparent;
    const agentlessConfig = (_appContextService$ge3 = _app_context.appContextService.getConfig()) === null || _appContextService$ge3 === void 0 ? void 0 : _appContextService$ge3.agentless;
    const kibanaVersion = _app_context.appContextService.getKibanaVersion();
    const tlsConfig = this.createTlsConfig(agentlessConfig);
    const urlEndpoint = (0, _agentless.prependAgentlessApiBasePathToEndpoint)(agentlessConfig, `/deployments/${policyId}`).split('/api')[1];
    logger.info(`[Agentless API] Call Agentless API endpoint ${urlEndpoint} to upgrade agentless deployment`);
    const requestConfig = {
      url: (0, _agentless.prependAgentlessApiBasePathToEndpoint)(agentlessConfig, `/deployments/${policyId}`),
      method: 'PUT',
      data: {
        stack_version: kibanaVersion
      },
      ...this.getHeaders(tlsConfig, traceId)
    };
    const errorMetadata = {
      trace: {
        id: traceId
      }
    };
    const requestConfigDebugStatus = this.createRequestConfigDebug(requestConfig);
    logger.info(`[Agentless API] Start upgrading agentless deployment for agent policy ${requestConfigDebugStatus}`);
    if (!_agentless.isAgentlessEnabled) {
      logger.error('[Agentless API] Agentless API is not supported. Upgrading agentless agent is not supported in non-cloud');
    }
    if (!agentlessConfig) {
      logger.error('[Agentless API] kibana.yml is currently missing Agentless API configuration');
    }
    logger.info(`[Agentless API] Upgrading agentless agent with TLS config with certificate`);
    logger.info(`[Agentless API] Upgrade agentless deployment with request config ${requestConfigDebugStatus}`);
    const response = await (0, _axios.default)(requestConfig).catch(async error => {
      await this.handleErrorsWithRetries(error, requestConfig, 'upgrade', logger, _agentless2.MAXIMUM_RETRIES, policyId, requestConfigDebugStatus, errorMetadata, traceId);
    });
    return response;
  }
  getHeaders(tlsConfig, traceId) {
    return {
      headers: {
        'Content-type': 'application/json',
        'X-Request-ID': traceId,
        'x-elastic-internal-origin': 'Kibana'
      },
      httpsAgent: new _https.default.Agent({
        rejectUnauthorized: tlsConfig.rejectUnauthorized,
        cert: tlsConfig.certificate,
        key: tlsConfig.key,
        ca: tlsConfig.certificateAuthorities
      })
    };
  }
  getAgentlessTags(agentlessAgentPolicy) {
    if (!agentlessAgentPolicy.global_data_tags) {
      return undefined;
    }
    const getGlobalTagValueByName = name => {
      var _agentlessAgentPolicy2, _agentlessAgentPolicy3;
      return (_agentlessAgentPolicy2 = agentlessAgentPolicy.global_data_tags) === null || _agentlessAgentPolicy2 === void 0 ? void 0 : (_agentlessAgentPolicy3 = _agentlessAgentPolicy2.find(tag => tag.name === name)) === null || _agentlessAgentPolicy3 === void 0 ? void 0 : _agentlessAgentPolicy3.value;
    };
    return {
      owner: {
        org: getGlobalTagValueByName(_constants.AGENTLESS_GLOBAL_TAG_NAME_ORGANIZATION),
        division: getGlobalTagValueByName(_constants.AGENTLESS_GLOBAL_TAG_NAME_DIVISION),
        team: getGlobalTagValueByName(_constants.AGENTLESS_GLOBAL_TAG_NAME_TEAM)
      }
    };
  }
  withRequestIdMessage(message, traceId) {
    return `${message} [Request Id: ${traceId}]`;
  }
  createTlsConfig(agentlessConfig) {
    var _agentlessConfig$api6, _agentlessConfig$api7, _agentlessConfig$api8, _agentlessConfig$api9, _agentlessConfig$api10, _agentlessConfig$api11;
    return new _serverHttpTools.SslConfig(_serverHttpTools.sslSchema.validate({
      enabled: true,
      certificate: agentlessConfig === null || agentlessConfig === void 0 ? void 0 : (_agentlessConfig$api6 = agentlessConfig.api) === null || _agentlessConfig$api6 === void 0 ? void 0 : (_agentlessConfig$api7 = _agentlessConfig$api6.tls) === null || _agentlessConfig$api7 === void 0 ? void 0 : _agentlessConfig$api7.certificate,
      key: agentlessConfig === null || agentlessConfig === void 0 ? void 0 : (_agentlessConfig$api8 = agentlessConfig.api) === null || _agentlessConfig$api8 === void 0 ? void 0 : (_agentlessConfig$api9 = _agentlessConfig$api8.tls) === null || _agentlessConfig$api9 === void 0 ? void 0 : _agentlessConfig$api9.key,
      certificateAuthorities: agentlessConfig === null || agentlessConfig === void 0 ? void 0 : (_agentlessConfig$api10 = agentlessConfig.api) === null || _agentlessConfig$api10 === void 0 ? void 0 : (_agentlessConfig$api11 = _agentlessConfig$api10.tls) === null || _agentlessConfig$api11 === void 0 ? void 0 : _agentlessConfig$api11.ca
    }));
  }
  async getFleetUrlAndTokenForAgentlessAgent(esClient, policyId, soClient) {
    const {
      items: enrollmentApiKeys
    } = await (0, _api_keys.listEnrollmentApiKeys)(esClient, {
      perPage: _constants.SO_SEARCH_LIMIT,
      showInactive: true,
      kuery: `policy_id:"${policyId}"`
    });
    const {
      items: fleetHosts
    } = await (0, _fleet_server_host.listFleetServerHosts)(soClient);
    // Tech Debt: change this when we add the internal fleet server config to use the internal fleet server host
    // https://github.com/elastic/security-team/issues/9695
    const defaultFleetHost = fleetHosts.length === 1 ? fleetHosts[0] : fleetHosts.find(host => host.is_default);
    if (!defaultFleetHost) {
      throw new _errors.AgentlessAgentConfigError('missing default Fleet server host');
    }
    if (!enrollmentApiKeys.length) {
      throw new _errors.AgentlessAgentConfigError('missing Fleet enrollment token');
    }
    const fleetToken = enrollmentApiKeys[0].api_key;
    const fleetUrl = defaultFleetHost === null || defaultFleetHost === void 0 ? void 0 : defaultFleetHost.host_urls[0];
    return {
      fleetUrl,
      fleetToken
    };
  }
  createRequestConfigDebug(requestConfig) {
    return JSON.stringify({
      ...requestConfig,
      data: {
        ...requestConfig.data,
        fleet_token: '[REDACTED]'
      },
      httpsAgent: {
        ...requestConfig.httpsAgent,
        options: {
          ...requestConfig.httpsAgent.options,
          cert: requestConfig.httpsAgent.options.cert ? 'REDACTED' : undefined,
          key: requestConfig.httpsAgent.options.key ? 'REDACTED' : undefined,
          ca: requestConfig.httpsAgent.options.ca ? 'REDACTED' : undefined
        }
      }
    });
  }
  catchAgentlessApiError(action, error, logger, agentlessPolicyId, requestConfig, requestConfigDebugStatus, errorMetadata, traceId) {
    const errorMetadataWithRequestConfig = {
      ...errorMetadata,
      http: {
        request: {
          id: traceId,
          body: requestConfig.data
        }
      }
    };
    const errorLogCodeCause = axiosError => `${axiosError.code}  ${this.convertCauseErrorsToString(axiosError)}`;
    if (!_axios.default.isAxiosError(error)) {
      let errorLogMessage;
      if (action === 'create') {
        errorLogMessage = `[Agentless API] Creating agentless failed with an error that is not an AxiosError for agentless policy`;
      }
      if (action === 'delete') {
        errorLogMessage = `[Agentless API] Deleting agentless deployment failed with an error that is not an Axios error for agentless policy`;
      }
      if (action === 'upgrade') {
        errorLogMessage = `[Agentless API] Upgrading agentless deployment failed with an error that is not an Axios error for agentless policy`;
      }
      logger.error(`${errorLogMessage} ${error} ${requestConfigDebugStatus}`, errorMetadataWithRequestConfig);
      throw this.getAgentlessAgentError(action, error.message, traceId);
    }
    const ERROR_HANDLING_MESSAGES = this.getErrorHandlingMessages(agentlessPolicyId);
    if (error.response) {
      if (error.response.status in ERROR_HANDLING_MESSAGES) {
        const handledResponseErrorMessage = ERROR_HANDLING_MESSAGES[error.response.status][action];
        this.handleResponseError(action, error.response, logger, errorMetadataWithRequestConfig, requestConfigDebugStatus, handledResponseErrorMessage.log, handledResponseErrorMessage.message, traceId);
      } else {
        const unhandledResponseErrorMessage = ERROR_HANDLING_MESSAGES.unhandled_response[action];
        // The request was made and the server responded with a status code and error data
        this.handleResponseError(action, error.response, logger, errorMetadataWithRequestConfig, requestConfigDebugStatus, unhandledResponseErrorMessage.log, unhandledResponseErrorMessage.message, traceId);
      }
    } else if (error.request) {
      // The request was made but no response was received
      const requestErrorMessage = ERROR_HANDLING_MESSAGES.request_error[action];
      logger.error(`${requestErrorMessage.log} ${errorLogCodeCause(error)} ${requestConfigDebugStatus}`, errorMetadataWithRequestConfig);
      throw this.getAgentlessAgentError(action, requestErrorMessage.message, traceId);
    } else {
      // Something happened in setting up the request that triggered an Error
      logger.error(`[Agentless API] ${action + 'ing'} the agentless agent failed ${errorLogCodeCause(error)} ${requestConfigDebugStatus}`, errorMetadataWithRequestConfig);
      throw this.getAgentlessAgentError(action, `the Agentless API could not ${action} the agentless agent`, traceId);
    }
  }
  handleResponseError(action, response, logger, errorMetadataWithRequestConfig, requestConfigDebugStatus, logMessage, userMessage, traceId) {
    logger.error(`${logMessage} ${JSON.stringify(response.status)} ${JSON.stringify(response.data)}} ${requestConfigDebugStatus}`, {
      ...errorMetadataWithRequestConfig,
      http: {
        ...errorMetadataWithRequestConfig.http,
        response: {
          status_code: response === null || response === void 0 ? void 0 : response.status,
          body: response === null || response === void 0 ? void 0 : response.data
        }
      }
    });
    throw this.getAgentlessAgentError(action, userMessage, traceId);
  }
  getAgentlessAgentError(action, userMessage, traceId) {
    if (action === 'create') {
      return new _errors.AgentlessAgentCreateError(this.withRequestIdMessage(userMessage, traceId));
    }
    if (action === 'delete') {
      return new _errors.AgentlessAgentDeleteError(this.withRequestIdMessage(userMessage, traceId));
    }
    if (action === 'upgrade') {
      return new _errors.AgentlessAgentUpgradeError(this.withRequestIdMessage(userMessage, traceId));
    }
  }
  getErrorHandlingMessages(agentlessPolicyId) {
    return {
      400: {
        create: {
          log: '[Agentless API] Creating the agentless agent failed with a status 400, bad request for agentless policy.',
          message: `The Agentless API could not create the agentless agent. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        },
        delete: {
          log: '[Agentless API] Deleting the agentless deployment failed with a status 400, bad request for agentless policy.',
          message: `The Agentless API could not delete the agentless deployment. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        },
        upgrade: {
          log: '[Agentless API] Upgrading the agentless agent failed with a status 400, bad request for agentless policy.',
          message: `The Agentless API could not upgrade the agentless agent. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        }
      },
      401: {
        create: {
          log: '[Agentless API] Creating the agentless agent failed with a status 401 unauthorized for agentless policy.',
          message: `The Agentless API could not create the agentless agent because an unauthorized request was sent. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        },
        delete: {
          log: '[Agentless API] Deleting the agentless deployment failed with a status 401 unauthorized for agentless policy.',
          message: `The Agentless API could not delete the agentless deployment because an unauthorized request was sent. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        },
        upgrade: {
          log: '[Agentless API] Upgrading the agentless agent failed with a status 401 unauthorized for agentless policy.',
          message: `The Agentless API could not upgrade the agentless agent because an unauthorized request was sent. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        }
      },
      403: {
        create: {
          log: '[Agentless API] Creating the agentless agent failed with a status 403 forbidden for agentless policy.',
          message: `The Agentless API could not create the agentless agent because a forbidden request was sent. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        },
        delete: {
          log: '[Agentless API] Deleting the agentless deployment failed with a status 403 forbidden for agentless policy.',
          message: `The Agentless API could not delete the agentless deployment because a forbidden request was sent. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        },
        upgrade: {
          log: '[Agentless API] Upgrading the agentless agent failed with a status 403 forbidden for agentless policy.',
          message: `The Agentless API could not upgrade the agentless agent because a forbidden request was sent. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        }
      },
      404: {
        create: {
          log: '[Agentless API] Creating the agentless agent failed with a status 404 not found.',
          message: `The Agentless API could not create the agentless agent because it returned a 404 error. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        },
        delete: {
          log: '[Agentless API] Deleting the agentless deployment failed with a status 404 not found.',
          message: `The Agentless API could not delete the agentless deployment because it could not be found. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        },
        upgrade: {
          log: '[Agentless API] Upgrading the agentless agent failed with a status 404 not found.',
          message: `The Agentless API could not upgrade the agentless agent because it returned a 404 error. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        }
      },
      408: {
        create: {
          log: '[Agentless API] Creating the agentless agent failed with a status 408, the request timed out.',
          message: `The Agentless API request timed out. Please wait a few minutes for the agent to enroll with Fleet. If the agent fails to enroll, delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        },
        delete: {
          log: '[Agentless API] Deleting the agentless deployment failed with a status 408, the request timed out.',
          message: `The Agentless API request timed out. Please wait a few minutes for the deployment to be removed. If it persists, delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        },
        upgrade: {
          log: '[Agentless API] Upgrading the agentless agent failed with a status 408, the request timed out.',
          message: `The Agentless API request timed out during the upgrade process. Please try again later or contact your administrator.`
        }
      },
      429: {
        create: {
          log: '[Agentless API] Creating the agentless agent failed with a status 429, agentless agent limit reached.',
          message: 'You have reached the limit for agentless provisioning. Please remove some or switch to agent-based integration.'
        },
        upgrade: {
          log: '[Agentless API] Upgrading the agentless agent failed with a status 429, agentless agent limit reached.',
          message: 'You have reached the limit for agentless provisioning. Please remove some or switch to agent-based integration.'
        }
      },
      500: {
        create: {
          log: '[Agentless API] Creating the agentless agent failed with a status 500 internal service error.',
          message: `The Agentless API could not create the agentless agent because it returned a 500 error. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        },
        delete: {
          log: '[Agentless API] Deleting the agentless deployment failed with a status 500 internal service error.',
          message: `The Agentless API could not delete the agentless deployment because it returned a 500 error. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        },
        upgrade: {
          log: '[Agentless API] Upgrading the agentless agent failed with a status 500 internal service error.',
          message: `The Agentless API could not upgrade the agentless agent because it returned a 500 error. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        }
      },
      unhandled_response: {
        create: {
          log: '[Agentless API] Creating the agentless agent failed with an unhandled response.',
          message: `The Agentless API could not create the agentless agent due to an unexpected error. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        },
        delete: {
          log: '[Agentless API] Deleting the agentless deployment failed with an unhandled response.',
          message: `The Agentless API could not delete the agentless deployment due to an unexpected error. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        },
        upgrade: {
          log: '[Agentless API] Upgrading the agentless agent failed with an unhandled response.',
          message: `The Agentless API could not upgrade the agentless agent due to an unexpected error. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        }
      },
      request_error: {
        create: {
          log: '[Agentless API] Creating the agentless agent failed with a request error.',
          message: `The Agentless API could not create the agentless agent due to a request error. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        },
        delete: {
          log: '[Agentless API] Deleting the agentless deployment failed with a request error.',
          message: `The Agentless API could not delete the agentless deployment due to a request error. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        },
        upgrade: {
          log: '[Agentless API] Upgrading the agentless agent failed with a request error.',
          message: `The Agentless API could not upgrade the agentless agent due to a request error. Please delete the agentless policy ${agentlessPolicyId} and try again or contact your administrator.`
        }
      }
    };
  }
}
const agentlessAgentService = exports.agentlessAgentService = new AgentlessAgentService();