"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EntityClient = void 0;
var _install_entity_definition = require("./entities/install_entity_definition");
var _start_transforms = require("./entities/start_transforms");
var _find_entity_definition = require("./entities/find_entity_definition");
var _uninstall_entity_definition = require("./entities/uninstall_entity_definition");
var _entity_not_found = require("./entities/errors/entity_not_found");
var _stop_transforms = require("./entities/stop_transforms");
var _delete_index = require("./entities/delete_index");
var _entity_definition_update_conflict = require("./entities/errors/entity_definition_update_conflict");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class EntityClient {
  constructor(options) {
    this.options = options;
  }
  async createEntityDefinition({
    definition,
    installOnly = false
  }) {
    this.options.logger.info(`Creating definition [${definition.id}] v${definition.version} (installOnly=${installOnly})`);
    const installedDefinition = await (0, _install_entity_definition.installEntityDefinition)({
      definition,
      esClient: this.options.clusterClient.asCurrentUser,
      soClient: this.options.soClient,
      logger: this.options.logger
    });
    if (!installOnly) {
      await (0, _start_transforms.startTransforms)(this.options.clusterClient.asCurrentUser, installedDefinition, this.options.logger);
    }
    return installedDefinition;
  }
  async updateEntityDefinition({
    id,
    definitionUpdate
  }) {
    const definition = await (0, _find_entity_definition.findEntityDefinitionById)({
      id,
      soClient: this.options.soClient,
      esClient: this.options.clusterClient.asCurrentUser,
      includeState: true
    });
    if (!definition) {
      const message = `Unable to find entity definition [${id}]`;
      this.options.logger.error(message);
      throw new _entity_not_found.EntityDefinitionNotFound(message);
    }
    if ((0, _install_entity_definition.installationInProgress)(definition)) {
      const message = `Entity definition [${definition.id}] has changes in progress`;
      this.options.logger.error(message);
      throw new _entity_definition_update_conflict.EntityDefinitionUpdateConflict(message);
    }
    const shouldRestartTransforms = definition.state.components.transforms.some(transform => transform.running);
    this.options.logger.info(`Updating definition [${definition.id}] from v${definition.version} to v${definitionUpdate.version}`);
    const updatedDefinition = await (0, _install_entity_definition.reinstallEntityDefinition)({
      definition,
      definitionUpdate,
      soClient: this.options.soClient,
      esClient: this.options.clusterClient.asCurrentUser,
      logger: this.options.logger
    });
    if (shouldRestartTransforms) {
      await (0, _start_transforms.startTransforms)(this.options.clusterClient.asCurrentUser, updatedDefinition, this.options.logger);
    }
    return updatedDefinition;
  }
  async deleteEntityDefinition({
    id,
    deleteData = false
  }) {
    const definition = await (0, _find_entity_definition.findEntityDefinitionById)({
      id,
      esClient: this.options.clusterClient.asCurrentUser,
      soClient: this.options.soClient
    });
    if (!definition) {
      throw new _entity_not_found.EntityDefinitionNotFound(`Unable to find entity definition [${id}]`);
    }
    this.options.logger.info(`Uninstalling definition [${definition.id}] v${definition.version} (deleteData=${deleteData})`);
    await (0, _uninstall_entity_definition.uninstallEntityDefinition)({
      definition,
      esClient: this.options.clusterClient.asCurrentUser,
      soClient: this.options.soClient,
      logger: this.options.logger
    });
    if (deleteData) {
      await (0, _delete_index.deleteIndices)(this.options.clusterClient.asCurrentUser, definition, this.options.logger);
    }
  }
  async getEntityDefinitions({
    id,
    page = 1,
    perPage = 10,
    includeState = false,
    type,
    builtIn
  }) {
    const definitions = await (0, _find_entity_definition.findEntityDefinitions)({
      esClient: this.options.clusterClient.asCurrentUser,
      soClient: this.options.soClient,
      page,
      perPage,
      id,
      includeState,
      type,
      builtIn
    });
    return {
      definitions
    };
  }
  async startEntityDefinition(definition) {
    this.options.logger.info(`Starting transforms for definition [${definition.id}]`);
    return (0, _start_transforms.startTransforms)(this.options.clusterClient.asCurrentUser, definition, this.options.logger);
  }
  async stopEntityDefinition(definition) {
    this.options.logger.info(`Stopping transforms for definition [${definition.id}]`);
    return (0, _stop_transforms.stopTransforms)(this.options.clusterClient.asCurrentUser, definition, this.options.logger);
  }
}
exports.EntityClient = EntityClient;