"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.splitResultFields = exports.resetAllFields = exports.convertToServerFieldResultSetting = exports.convertServerResultFieldsToResultFields = exports.clearAllFields = exports.areFieldsEmpty = exports.areFieldsAtDefaultSettings = void 0;
var _lodash = require("lodash");
var _types = require("../../../shared/schema/types");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const updateAllFields = (fields, newValue) => {
  return Object.keys(fields).reduce((acc, fieldName) => ({
    ...acc,
    [fieldName]: {
      ...newValue
    }
  }), {});
};
const convertToFieldResultSetting = serverFieldResultSetting => {
  const fieldResultSetting = {
    raw: !!serverFieldResultSetting.raw,
    snippet: !!serverFieldResultSetting.snippet,
    snippetFallback: !!(serverFieldResultSetting.snippet && typeof serverFieldResultSetting.snippet === 'object' && serverFieldResultSetting.snippet.fallback)
  };
  if (serverFieldResultSetting.raw && typeof serverFieldResultSetting.raw === 'object' && serverFieldResultSetting.raw.size) {
    fieldResultSetting.rawSize = serverFieldResultSetting.raw.size;
  }
  if (serverFieldResultSetting.snippet && typeof serverFieldResultSetting.snippet === 'object' && serverFieldResultSetting.snippet.size) {
    fieldResultSetting.snippetSize = serverFieldResultSetting.snippet.size;
  }
  return fieldResultSetting;
};
const clearAllFields = fields => updateAllFields(fields, {});
exports.clearAllFields = clearAllFields;
const resetAllFields = fields => updateAllFields(fields, _constants.DEFAULT_FIELD_SETTINGS);
exports.resetAllFields = resetAllFields;
const convertServerResultFieldsToResultFields = (serverResultFields, schema) => {
  const resultFields = Object.keys(schema).reduce((acc, fieldName) => ({
    ...acc,
    [fieldName]: serverResultFields[fieldName] ? convertToFieldResultSetting(serverResultFields[fieldName]) : _constants.DISABLED_FIELD_SETTINGS
  }), {});
  return resultFields;
};
exports.convertServerResultFieldsToResultFields = convertServerResultFieldsToResultFields;
const convertToServerFieldResultSetting = fieldResultSetting => {
  const serverFieldResultSetting = {};
  if (fieldResultSetting.raw) {
    serverFieldResultSetting.raw = {};
    if (fieldResultSetting.rawSize) {
      serverFieldResultSetting.raw.size = fieldResultSetting.rawSize;
    }
  }
  if (fieldResultSetting.snippet) {
    serverFieldResultSetting.snippet = {};
    if (fieldResultSetting.snippetFallback) {
      serverFieldResultSetting.snippet.fallback = fieldResultSetting.snippetFallback;
    }
    if (fieldResultSetting.snippetSize) {
      serverFieldResultSetting.snippet.size = fieldResultSetting.snippetSize;
    }
  }
  return serverFieldResultSetting;
};
exports.convertToServerFieldResultSetting = convertToServerFieldResultSetting;
const splitResultFields = (resultFields, schema) => Object.entries(resultFields).reduce((acc, [fieldName, resultFieldSettings]) => {
  const fieldType = schema[fieldName].type;
  const targetField = fieldType === _types.SchemaType.Text ? 'textResultFields' : 'nonTextResultFields';
  return {
    ...acc,
    [targetField]: {
      ...acc[targetField],
      [fieldName]: resultFieldSettings
    }
  };
}, {
  textResultFields: {},
  nonTextResultFields: {}
});
exports.splitResultFields = splitResultFields;
const areFieldsEmpty = fields => {
  const anyNonEmptyField = Object.values(fields).find(field => {
    return field.raw || field.snippet;
  });
  return !anyNonEmptyField;
};
exports.areFieldsEmpty = areFieldsEmpty;
const areFieldsAtDefaultSettings = fields => {
  const anyNonDefaultSettingsValue = Object.values(fields).find(resultSettings => {
    return !(0, _lodash.isEqual)(resultSettings, _constants.DEFAULT_FIELD_SETTINGS);
  });
  return !anyNonDefaultSettingsValue;
};
exports.areFieldsAtDefaultSettings = areFieldsAtDefaultSettings;