"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EnginesLogic = void 0;
var _kea = require("kea");
var _constants = require("../../../shared/constants");
var _flash_messages = require("../../../shared/flash_messages");
var _http = require("../../../shared/http");
var _table_pagination = require("../../../shared/table_pagination");
var _types = require("../engine/types");
var _constants2 = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const EnginesLogic = exports.EnginesLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'engines_logic'],
  actions: {
    deleteEngine: engine => ({
      engine
    }),
    onDeleteEngineSuccess: engine => ({
      engine
    }),
    onEnginesLoad: ({
      results,
      meta
    }) => ({
      results,
      meta
    }),
    onMetaEnginesLoad: ({
      results,
      meta
    }) => ({
      results,
      meta
    }),
    onEnginesPagination: page => ({
      page
    }),
    onMetaEnginesPagination: page => ({
      page
    }),
    loadEngines: true,
    loadMetaEngines: true
  },
  reducers: {
    engines: [[], {
      onEnginesLoad: (_, {
        results
      }) => results
    }],
    enginesMeta: [_constants.DEFAULT_META, {
      onEnginesLoad: (_, {
        meta
      }) => meta,
      onEnginesPagination: (state, {
        page
      }) => (0, _table_pagination.updateMetaPageIndex)(state, page)
    }],
    enginesLoading: [true, {
      loadEngines: () => true,
      onEnginesLoad: () => false
    }],
    metaEngines: [[], {
      onMetaEnginesLoad: (_, {
        results
      }) => results
    }],
    metaEnginesMeta: [_constants.DEFAULT_META, {
      onMetaEnginesLoad: (_, {
        meta
      }) => meta,
      onMetaEnginesPagination: (state, {
        page
      }) => (0, _table_pagination.updateMetaPageIndex)(state, page)
    }],
    metaEnginesLoading: [true, {
      loadMetaEngines: () => true,
      onMetaEnginesLoad: () => false
    }]
  },
  selectors: {
    dataLoading: [selectors => [selectors.enginesLoading, selectors.engines], (enginesLoading, engines) => enginesLoading && !engines.length]
  },
  listeners: ({
    actions,
    values
  }) => ({
    deleteEngine: async ({
      engine
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      let response;
      try {
        response = await http.delete(`/internal/app_search/engines/${engine.name}`);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
      if (response) {
        actions.onDeleteEngineSuccess(engine);
      }
    },
    loadEngines: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        enginesMeta
      } = values;
      try {
        const response = await http.get('/internal/app_search/engines', {
          query: {
            type: 'indexed',
            'page[current]': enginesMeta.page.current,
            'page[size]': enginesMeta.page.size
          }
        });
        actions.onEnginesLoad(response);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    loadMetaEngines: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        metaEnginesMeta
      } = values;
      try {
        const response = await http.get('/internal/app_search/engines', {
          query: {
            type: 'meta',
            'page[current]': metaEnginesMeta.page.current,
            'page[size]': metaEnginesMeta.page.size
          }
        });
        actions.onMetaEnginesLoad(response);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    onDeleteEngineSuccess: async ({
      engine
    }) => {
      (0, _flash_messages.flashSuccessToast)((0, _constants2.DELETE_ENGINE_MESSAGE)(engine.name));
      if ([_types.EngineTypes.default, _types.EngineTypes.indexed, _types.EngineTypes.elasticsearch].includes(engine.type)) {
        actions.loadEngines();
      } else if (engine.type === _types.EngineTypes.meta) {
        actions.loadMetaEngines();
      }
    }
  })
});