"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createKibanaDataView = createKibanaDataView;
exports.getInferenceId = getInferenceId;
exports.importData = importData;
var _i18n = require("@kbn/i18n");
var _import_progress = require("../import_progress/import_progress");
var _auto_deploy = require("./auto_deploy");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function importData(props, config, setState) {
  const {
    data,
    results,
    dataViewsContract,
    fileUpload,
    http
  } = props;
  const {
    index,
    dataView,
    createDataView,
    indexSettingsString,
    mappingsString,
    pipelineString,
    pipelineId,
    createPipeline
  } = config;
  const {
    format
  } = results;
  const errors = [];
  if (index === '') {
    return;
  }
  if ((await fileUpload.hasImportPermission({
    checkCreateDataView: createDataView,
    checkHasManagePipeline: true,
    indexName: index
  })) === false) {
    errors.push(_i18n.i18n.translate('xpack.dataVisualizer.file.importView.importPermissionError', {
      defaultMessage: 'You do not have permission to create or import data into index {index}.',
      values: {
        index
      }
    }));
    setState({
      permissionCheckStatus: _import_progress.IMPORT_STATUS.FAILED,
      importing: false,
      imported: false,
      errors
    });
    return;
  }
  let success = true;
  let settings = {};
  let mappings = {};
  const pipelines = [];
  try {
    settings = JSON.parse(indexSettingsString);
  } catch (error) {
    success = false;
    const parseError = _i18n.i18n.translate('xpack.dataVisualizer.file.importView.parseSettingsError', {
      defaultMessage: 'Error parsing settings:'
    });
    errors.push(`${parseError} ${error.message}`);
  }
  try {
    mappings = JSON.parse(mappingsString);
  } catch (error) {
    success = false;
    const parseError = _i18n.i18n.translate('xpack.dataVisualizer.file.importView.parseMappingsError', {
      defaultMessage: 'Error parsing mappings:'
    });
    errors.push(`${parseError} ${error.message}`);
  }
  try {
    if (createPipeline) {
      pipelines.push(JSON.parse(pipelineString));
    }
  } catch (error) {
    success = false;
    const parseError = _i18n.i18n.translate('xpack.dataVisualizer.file.importView.parsePipelineError', {
      defaultMessage: 'Error parsing ingest pipeline:'
    });
    errors.push(`${parseError} ${error.message}`);
  }
  const inferenceId = getInferenceId(mappings);
  const initializeDeployment = _auto_deploy.AutoDeploy.shouldAutoDeploy(inferenceId);
  setState({
    importing: true,
    imported: false,
    reading: true,
    initialized: true,
    permissionCheckStatus: _import_progress.IMPORT_STATUS.COMPLETE,
    initializeDeployment,
    parseJSONStatus: getSuccess(success)
  });
  if (success === false) {
    return;
  }
  const importer = await fileUpload.importerFactory(format, {
    excludeLinesPattern: results.exclude_lines_pattern,
    multilineStartPattern: results.multiline_start_pattern
  });
  const readResp = importer.read(data);
  success = readResp.success;
  setState({
    readStatus: getSuccess(success),
    reading: false,
    importer
  });
  if (success === false) {
    return;
  }
  if (initializeDeployment) {
    // Initialize deployment
    const autoDeploy = new _auto_deploy.AutoDeploy(http, inferenceId);
    try {
      await autoDeploy.deploy();
      setState({
        initializeDeploymentStatus: _import_progress.IMPORT_STATUS.COMPLETE,
        inferenceId
      });
    } catch (error) {
      success = false;
      const deployError = _i18n.i18n.translate('xpack.dataVisualizer.file.importView.deployModelError', {
        defaultMessage: 'Error deploying trained model:'
      });
      errors.push(`${deployError} ${error.message}`);
      setState({
        initializeDeploymentStatus: _import_progress.IMPORT_STATUS.FAILED,
        errors
      });
    }
  }
  if (success === false) {
    return;
  }
  const initializeImportResp = await importer.initializeImport(index, settings, mappings, pipelines);
  if (initializeImportResp.success === false) {
    errors.push(initializeImportResp.error);
    setState({
      initializeImportStatus: _import_progress.IMPORT_STATUS.FAILED,
      errors
    });
    return;
  }
  const timeFieldName = importer.getTimeField();
  setState({
    timeFieldName
  });
  const indexCreated = initializeImportResp.index !== undefined;
  setState({
    indexCreatedStatus: getSuccess(indexCreated)
  });
  if (createPipeline) {
    const pipelinesCreated = initializeImportResp.pipelineIds.length > 0;
    if (indexCreated) {
      setState({
        ingestPipelineCreatedStatus: pipelinesCreated ? _import_progress.IMPORT_STATUS.COMPLETE : _import_progress.IMPORT_STATUS.FAILED,
        pipelineId: pipelinesCreated ? initializeImportResp.pipelineIds[0] : ''
      });
    }
    success = indexCreated && pipelinesCreated;
  } else {
    success = indexCreated;
  }
  if (success === false) {
    errors.push(initializeImportResp.error);
    return;
  }
  const importResp = await importer.import(index, pipelineId !== null && pipelineId !== void 0 ? pipelineId : initializeImportResp.pipelineIds[0], progress => {
    setState({
      uploadProgress: progress
    });
  });
  success = importResp.success;
  setState({
    uploadStatus: getSuccess(importResp.success),
    importFailures: importResp.failures,
    docCount: importResp.docCount
  });
  if (success === false) {
    errors.push(importResp.error);
    return;
  }
  if (createDataView) {
    const dataViewName = dataView === '' ? index : dataView;
    const dataViewResp = await createKibanaDataView(dataViewName, dataViewsContract, timeFieldName);
    success = dataViewResp.success;
    setState({
      dataViewCreatedStatus: dataViewResp.success ? _import_progress.IMPORT_STATUS.COMPLETE : _import_progress.IMPORT_STATUS.FAILED,
      dataViewId: dataViewResp.id
    });
    if (success === false) {
      errors.push(dataViewResp.error);
    }
  }
  setState({
    importing: false,
    imported: success,
    errors
  });
}
async function createKibanaDataView(dataViewName, dataViewsContract, timeFieldName) {
  try {
    const emptyPattern = await dataViewsContract.createAndSave({
      title: dataViewName,
      timeFieldName
    });
    return {
      success: true,
      id: emptyPattern.id
    };
  } catch (error) {
    return {
      success: false,
      error
    };
  }
}
function getSuccess(success) {
  return success ? _import_progress.IMPORT_STATUS.COMPLETE : _import_progress.IMPORT_STATUS.FAILED;
}
function getInferenceId(mappings) {
  for (const value of Object.values((_mappings$properties = mappings.properties) !== null && _mappings$properties !== void 0 ? _mappings$properties : {})) {
    var _mappings$properties;
    if (value.type === 'semantic_text' && value.inference_id) {
      return value.inference_id;
    }
  }
  return null;
}