"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createScriptedFieldsDeprecationsConfig = void 0;
exports.hasScriptedField = hasScriptedField;
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const createScriptedFieldsDeprecationsConfig = core => ({
  getDeprecations: async context => {
    const finder = context.savedObjectsClient.createPointInTimeFinder({
      type: 'index-pattern',
      perPage: 1000,
      fields: ['name', 'title', 'fields'],
      namespaces: ['*']
    });
    const dataViewsWithScriptedFields = [];
    for await (const response of finder.find()) {
      dataViewsWithScriptedFields.push(...response.saved_objects.filter(so => hasScriptedField(so.attributes)));
    }
    if (!dataViewsWithScriptedFields.length) {
      return [];
    }
    return [{
      title: _i18n.i18n.translate('dataViews.deprecations.scriptedFieldsTitle', {
        defaultMessage: 'Found data views using scripted fields'
      }),
      message: buildMessage({
        dataViewsWithScriptedFields,
        docLinks: core.docLinks.links
      }),
      documentationUrl: core.docLinks.links.indexPatterns.migrateOffScriptedFields,
      deprecationType: 'feature',
      level: 'warning',
      // warning because it is not set in stone WHEN we remove scripted fields, hence this deprecation is not a blocker for 9.0 upgrade
      correctiveActions: {
        manualSteps: [_i18n.i18n.translate('dataViews.deprecations.scriptedFields.manualStepOneMessage', {
          defaultMessage: 'Navigate to Stack Management > Kibana > Data Views.'
        }), _i18n.i18n.translate('dataViews.deprecations.scriptedFields.manualStepTwoMessage', {
          defaultMessage: 'Update data views that have scripted fields to use runtime fields instead. In most cases, you will only need to change "return <value>;" to "emit(<value>);".',
          ignoreTag: true
        }), _i18n.i18n.translate('dataViews.deprecations.scriptedFields.manualStepThreeMessage', {
          defaultMessage: 'Alternatively, you can achieve similar functionality by computing values at query time using the Elasticsearch Query Language (ES|QL).'
        })]
      }
    }];
  }
});
exports.createScriptedFieldsDeprecationsConfig = createScriptedFieldsDeprecationsConfig;
function hasScriptedField(dataView) {
  if (dataView.fields) {
    try {
      return JSON.parse(dataView.fields).some(field => field === null || field === void 0 ? void 0 : field.scripted);
    } catch (e) {
      return false;
    }
  } else {
    return false;
  }
}
const dataViewIdLabel = _i18n.i18n.translate('dataViews.deprecations.scriptedFields.dataViewIdLabel', {
  defaultMessage: 'ID'
});
const dataViewNameLabel = _i18n.i18n.translate('dataViews.deprecations.scriptedFields.dataViewNameLabel', {
  defaultMessage: 'Name'
});
const dataViewSpacesLabel = _i18n.i18n.translate('dataViews.deprecations.scriptedFields.dataViewSpacesLabel', {
  defaultMessage: 'Spaces'
});
const buildDataViewsListEntry = so => {
  var _so$namespaces;
  return `- **${dataViewIdLabel}:** ${so.id}
  - **${dataViewNameLabel}:** ${so.attributes.name ? `!{tooltip[${so.attributes.name}](${so.attributes.title})}` : so.attributes.title}
  - **${dataViewSpacesLabel}:** ${(_so$namespaces = so.namespaces) === null || _so$namespaces === void 0 ? void 0 : _so$namespaces.join(', ')}`;
};
const buildMessage = ({
  dataViewsWithScriptedFields,
  docLinks
}) => ({
  type: 'markdown',
  content: _i18n.i18n.translate('dataViews.deprecations.scriptedFieldsMessage', {
    defaultMessage: `You have {numberOfDataViewsWithScriptedFields} {numberOfDataViewsWithScriptedFields, plural, one {data view} other {data views}} containing scripted fields. Scripted fields are deprecated and will be removed in the future.

The ability to create new scripted fields in the Data Views management page has been disabled in 9.0, and it is recommended to migrate to [runtime fields]({runtimeFieldsLink}) or the [Elasticsearch Query Language (ES|QL)]({esqlLink}) instead.

The following is a list of all data views with scripted fields and their associated spaces:
{dataViewsList}`,
    values: {
      numberOfDataViewsWithScriptedFields: dataViewsWithScriptedFields.length,
      runtimeFieldsLink: docLinks.indexPatterns.runtimeFields,
      esqlLink: docLinks.query.queryESQL,
      dataViewsList: dataViewsWithScriptedFields.map(buildDataViewsListEntry).join('\n')
    }
  })
});