"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.sqlSearchStrategyProvider = void 0;
var _rxjs = require("rxjs");
var _server = require("@kbn/kibana-utils-plugin/server");
var _report_search_error = require("../../report_search_error");
var _common = require("../../../../common");
var _request_utils = require("./request_utils");
var _response_utils = require("./response_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const sqlSearchStrategyProvider = (searchConfig, logger, useInternalUser = false) => {
  function cancelAsyncSearch(id, esClient) {
    const client = useInternalUser ? esClient.asInternalUser : esClient.asCurrentUser;
    return client.sql.deleteAsync({
      id
    });
  }
  function asyncSearch({
    id,
    ...request
  }, options, {
    esClient
  }) {
    const client = useInternalUser ? esClient.asInternalUser : esClient.asCurrentUser;
    const startTime = Date.now();
    const search = async () => {
      var _request$params, _headers, _meta, _meta$request;
      const {
        keep_cursor: keepCursor,
        ...params
      } = (_request$params = request.params) !== null && _request$params !== void 0 ? _request$params : {};
      let body;
      let headers;
      let meta;
      if (id) {
        var _params$format;
        ({
          body,
          headers,
          meta
        } = await client.sql.getAsync({
          format: (_params$format = params === null || params === void 0 ? void 0 : params.format) !== null && _params$format !== void 0 ? _params$format : 'json',
          ...(0, _request_utils.getDefaultAsyncGetParams)(searchConfig, options),
          id
        }, {
          ...options.transport,
          signal: options.abortSignal,
          meta: true
        }));
      } else {
        var _params$format2;
        ({
          headers,
          body,
          meta
        } = await client.sql.query({
          ...(0, _request_utils.getDefaultAsyncSubmitParams)(searchConfig, options),
          ...params,
          format: (_params$format2 = params.format) !== null && _params$format2 !== void 0 ? _params$format2 : 'json'
        }, {
          ...options.transport,
          signal: options.abortSignal,
          meta: true
        }));
      }
      if (!body.is_running && body.cursor && !keepCursor) {
        try {
          await client.sql.clearCursor({
            cursor: body.cursor
          });
        } catch (error) {
          logger.warn(`sql search: failed to clear cursor=${body.cursor} for async_search_id=${id}: ${error.message}`);
        }
      }
      return (0, _response_utils.toAsyncKibanaSearchResponse)(body, startTime, (_headers = headers) === null || _headers === void 0 ? void 0 : _headers.warning,
      // do not return requestParams on polling calls
      id ? undefined : (_meta = meta) === null || _meta === void 0 ? void 0 : (_meta$request = _meta.request) === null || _meta$request === void 0 ? void 0 : _meta$request.params);
    };
    const cancel = async () => {
      if (!id) return;
      try {
        await cancelAsyncSearch(id, esClient);
      } catch (e) {
        var _e$meta;
        // A 404 means either this search request does not exist, or that it is already cancelled
        if (((_e$meta = e.meta) === null || _e$meta === void 0 ? void 0 : _e$meta.statusCode) === 404) return;

        // Log all other (unexpected) error messages
        logger.error(`cancelSqlSearch error: ${e.message}`);
      }
    };
    return (0, _common.pollSearch)(search, cancel, {
      pollInterval: searchConfig.asyncSearch.pollInterval,
      ...options
    }).pipe((0, _rxjs.tap)(response => id = response.id), (0, _rxjs.catchError)(e => {
      throw (0, _report_search_error.getKbnSearchError)(e);
    }));
  }
  return {
    /**
     * @param request
     * @param options
     * @param deps `SearchStrategyDependencies`
     * @returns `Observable<IEsSearchResponse<any>>`
     * @throws `KbnSearchError`
     */
    search: (request, options, deps) => {
      logger.debug(() => `sql search: search request=${JSON.stringify(request)}`);
      return asyncSearch(request, options, deps);
    },
    /**
     * @param id async search ID to cancel, as returned from _async_search API
     * @param options
     * @param deps `SearchStrategyDependencies`
     * @returns `Promise<void>`
     * @throws `KbnServerError`
     */
    cancel: async (id, options, {
      esClient
    }) => {
      logger.debug(`sql search: cancel async_search_id=${id}`);
      try {
        await cancelAsyncSearch(id, esClient);
      } catch (e) {
        throw (0, _server.getKbnServerError)(e);
      }
    },
    /**
     *
     * @param id async search ID to extend, as returned from _async_search API
     * @param keepAlive
     * @param options
     * @param deps `SearchStrategyDependencies`
     * @returns `Promise<void>`
     * @throws `KbnServerError`
     */
    extend: async (id, keepAlive, options, {
      esClient
    }) => {
      logger.debug(`sql search: extend async_search_id=${id}  keep_alive=${keepAlive}`);
      try {
        const client = useInternalUser ? esClient.asInternalUser : esClient.asCurrentUser;
        await client.sql.getAsync({
          id,
          keep_alive: keepAlive
        });
      } catch (e) {
        throw (0, _server.getKbnServerError)(e);
      }
    }
  };
};
exports.sqlSearchStrategyProvider = sqlSearchStrategyProvider;