"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.loadDashboardState = void 0;
exports.migrateLegacyQuery = migrateLegacyQuery;
var _public = require("@kbn/data-plugin/public");
var _public2 = require("@kbn/kibana-utils-plugin/public");
var _lodash = require("lodash");
var _ = require("..");
var _default_dashboard_state = require("../../../dashboard_api/default_dashboard_state");
var _clean_filters_for_serialize = require("../../../utils/clean_filters_for_serialize");
var _telemetry_constants = require("../../../utils/telemetry_constants");
var _kibana_services = require("../../kibana_services");
var _dashboard_versioning = require("./dashboard_versioning");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function migrateLegacyQuery(query) {
  // Lucene was the only option before, so language-less queries are all lucene
  if (!(0, _lodash.has)(query, 'language')) {
    return {
      query,
      language: 'lucene'
    };
  }
  return query;
}
const loadDashboardState = async ({
  id
}) => {
  var _ref, _savedObjectsTaggingS, _savedObjectsTaggingS2;
  const {
    search: dataSearchService,
    query: {
      queryString
    }
  } = _kibana_services.dataService;
  const dashboardContentManagementCache = (0, _.getDashboardContentManagementCache)();
  const savedObjectId = id;
  const newDashboardState = {
    ..._default_dashboard_state.DEFAULT_DASHBOARD_STATE
  };

  /**
   * This is a newly created dashboard, so there is no saved object state to load.
   */
  if (!savedObjectId) {
    return {
      dashboardInput: newDashboardState,
      dashboardFound: true,
      newDashboardCreated: true,
      references: []
    };
  }

  /**
   * Load the saved object from Content Management
   */
  let rawDashboardContent;
  let resolveMeta;
  const cachedDashboard = dashboardContentManagementCache.fetchDashboard(id);
  if (cachedDashboard) {
    /** If the dashboard exists in the cache, use the cached version to load the dashboard */
    ({
      item: rawDashboardContent,
      meta: resolveMeta
    } = cachedDashboard);
  } else {
    /** Otherwise, fetch and load the dashboard from the content management client, and add it to the cache */
    const result = await _kibana_services.contentManagementService.client.get({
      contentTypeId: _telemetry_constants.DASHBOARD_CONTENT_ID,
      id
    }).catch(e => {
      var _e$response, _message, _e$body;
      if (((_e$response = e.response) === null || _e$response === void 0 ? void 0 : _e$response.status) === 404) {
        throw new _public2.SavedObjectNotFound(_telemetry_constants.DASHBOARD_CONTENT_ID, id);
      }
      const message = (_message = (_e$body = e.body) === null || _e$body === void 0 ? void 0 : _e$body.message) !== null && _message !== void 0 ? _message : e.message;
      throw new Error(message);
    });
    ({
      item: rawDashboardContent,
      meta: resolveMeta
    } = result);
    const {
      outcome: loadOutcome
    } = resolveMeta;
    if (loadOutcome !== 'aliasMatch') {
      /**
       * Only add the dashboard to the cache if it does not require a redirect - otherwise, the meta
       * alias info gets cached and prevents the dashboard contents from being updated
       */
      dashboardContentManagementCache.addDashboard(result);
    }
  }
  if (!rawDashboardContent || !rawDashboardContent.version) {
    return {
      dashboardInput: newDashboardState,
      dashboardFound: false,
      dashboardId: savedObjectId,
      references: []
    };
  }
  const {
    references,
    attributes,
    managed
  } = rawDashboardContent;

  /**
   * Create search source and pull filters and query from it.
   */
  let searchSourceValues = attributes.kibanaSavedObjectMeta.searchSource;
  const searchSource = await (async () => {
    if (!searchSourceValues) {
      return await dataSearchService.searchSource.create();
    }
    try {
      searchSourceValues = (0, _public.injectSearchSourceReferences)(searchSourceValues, references);
      return await dataSearchService.searchSource.create(searchSourceValues);
    } catch (error) {
      return await dataSearchService.searchSource.create();
    }
  })();
  const filters = (0, _clean_filters_for_serialize.cleanFiltersForSerialize)((_ref = searchSource === null || searchSource === void 0 ? void 0 : searchSource.getOwnField('filter')) !== null && _ref !== void 0 ? _ref : []);
  const query = migrateLegacyQuery((searchSource === null || searchSource === void 0 ? void 0 : searchSource.getOwnField('query')) || queryString.getDefaultQuery() // TODO SAVED DASHBOARDS determine if migrateLegacyQuery is still needed
  );
  const {
    refreshInterval,
    description,
    timeRestore,
    options,
    panels,
    timeFrom,
    version,
    timeTo,
    title
  } = attributes;
  const timeRange = timeRestore && timeFrom && timeTo ? {
    from: timeFrom,
    to: timeTo
  } : undefined;
  return {
    managed,
    references,
    resolveMeta,
    dashboardInput: {
      ..._default_dashboard_state.DEFAULT_DASHBOARD_STATE,
      ...options,
      refreshInterval,
      timeRestore,
      description,
      timeRange,
      filters,
      panels,
      query,
      title,
      viewMode: 'view',
      // dashboards loaded from saved object default to view mode. If it was edited recently, the view mode from session storage will override this.
      tags: (_savedObjectsTaggingS = _kibana_services.savedObjectsTaggingService === null || _kibana_services.savedObjectsTaggingService === void 0 ? void 0 : (_savedObjectsTaggingS2 = _kibana_services.savedObjectsTaggingService.getTaggingApi()) === null || _savedObjectsTaggingS2 === void 0 ? void 0 : _savedObjectsTaggingS2.ui.getTagIdsFromReferences(references)) !== null && _savedObjectsTaggingS !== void 0 ? _savedObjectsTaggingS : [],
      controlGroupInput: attributes.controlGroupInput,
      ...(version && {
        version: (0, _dashboard_versioning.convertNumberToDashboardVersion)(version)
      })
    },
    dashboardFound: true,
    dashboardId: savedObjectId
  };
};
exports.loadDashboardState = loadDashboardState;