"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.toScheduledItem = exports.findOverlappingIntervals = void 0;
var _common = require("../../../../common");
var _interval_utils = require("../gap/interval_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const toScheduledItem = backfillSchedule => {
  const runAt = new Date(backfillSchedule.runAt).getTime();
  const intervalDuration = (0, _common.parseDuration)(backfillSchedule.interval);
  const from = runAt - intervalDuration;
  const to = runAt;
  return {
    from: new Date(from),
    to: new Date(to),
    status: backfillSchedule.status
  };
};
exports.toScheduledItem = toScheduledItem;
const findEarliestOverlapping = (startMs, endMs, scheduledItems) => {
  let startIdx = 0;
  let endIdx = scheduledItems.length - 1;
  let earliestOverlapping = scheduledItems.length;
  while (startIdx <= endIdx) {
    const midIdx = Math.floor((endIdx - startIdx) / 2) + startIdx;
    const scheduledItem = scheduledItems[midIdx];
    if (endMs <= scheduledItem.from.getTime()) {
      endIdx = midIdx - 1;
      continue;
    }
    if (startMs >= scheduledItem.to.getTime()) {
      startIdx = midIdx + 1;
      continue;
    }

    // There is an overlap at this point
    earliestOverlapping = Math.min(earliestOverlapping, midIdx);
    // go left to see if there is an erlier interval
    endIdx = midIdx - 1;
  }
  return earliestOverlapping < scheduledItems.length ? earliestOverlapping : null;
};
const clipScheduled = (startMs, endMs, scheduledItem) => {
  const clipped = (0, _interval_utils.clipDateInterval)(scheduledItem.from, scheduledItem.to, new Date(startMs), new Date(endMs));
  if (clipped === null) {
    throw new Error('Clipped interval in an scheduled item cannot be null');
  }
  return {
    from: clipped.start,
    to: clipped.end,
    status: scheduledItem.status
  };
};
const findOverlapping = (startMs, endMs, scheduledItems) => {
  let earliestIdx = findEarliestOverlapping(startMs, endMs, scheduledItems);
  const overlapping = [];
  if (earliestIdx === null) {
    return overlapping;
  }
  while (earliestIdx < scheduledItems.length) {
    const scheduled = scheduledItems[earliestIdx];
    if (scheduled.from.getTime() >= endMs) {
      break;
    }
    overlapping.push(scheduled);
    earliestIdx++;
  }
  overlapping[0] = clipScheduled(startMs, endMs, overlapping[0]);
  overlapping[overlapping.length - 1] = clipScheduled(startMs, endMs, overlapping[overlapping.length - 1]);
  return overlapping;
};
const findOverlappingIntervals = (gaps, scheduledItems) => {
  return gaps.map(gap => {
    return {
      gap,
      scheduled: findOverlapping(gap.range.gte.getTime(), gap.range.lte.getTime(), scheduledItems)
    };
  });
};
exports.findOverlappingIntervals = findOverlappingIntervals;