"use strict";

var _react = _interopRequireDefault(require("react"));
var _faker = require("@faker-js/faker");
var _ = require(".");
var _health = require("../health");
var _link = require("../link");
var _services = require("../../services");
var _react2 = require("@emotion/react");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/// <reference types="cypress" />
/// <reference types="cypress-real-events" />
/// <reference types="../../../cypress/support" />

var users = [];
for (var i = 0; i < 20; i++) {
  users.push({
    id: i + 1,
    firstName: _faker.faker.person.firstName(),
    lastName: _faker.faker.person.lastName(),
    github: _faker.faker.internet.userName(),
    dateOfBirth: _faker.faker.date.past(),
    online: _faker.faker.datatype.boolean(),
    location: {
      city: _faker.faker.location.city(),
      country: _faker.faker.location.country()
    }
  });
}
var columns = [{
  field: 'firstName',
  name: 'First Name',
  sortable: true,
  truncateText: true,
  mobileOptions: {
    render: function render(user) {
      return (0, _react2.jsx)("span", null, user.firstName, " ", user.lastName);
    },
    header: false,
    truncateText: false,
    enlarge: true,
    width: '100%'
  }
}, {
  field: 'lastName',
  name: 'Last Name',
  truncateText: true,
  mobileOptions: {
    show: false
  }
}, {
  field: 'github',
  name: 'Github',
  render: function render(username) {
    return (0, _react2.jsx)(_link.EuiLink, {
      href: "#",
      target: "_blank"
    }, username);
  }
}, {
  field: 'dateOfBirth',
  name: 'Date of Birth',
  dataType: 'date',
  render: function render(dateOfBirth) {
    return (0, _services.formatDate)(dateOfBirth, 'dobLong');
  },
  sortable: true
}, {
  field: 'location',
  name: 'Location',
  truncateText: true,
  textOnly: true,
  render: function render(location) {
    return "".concat(location.city, ", ").concat(location.country);
  }
}, {
  field: 'online',
  name: 'Online',
  dataType: 'boolean',
  render: function render(online) {
    var color = online ? 'success' : 'danger';
    var label = online ? 'Online' : 'Offline';
    return (0, _react2.jsx)(_health.EuiHealth, {
      color: color
    }, label);
  },
  sortable: true,
  mobileOptions: {
    show: false
  }
}];
describe('EuiInMemoryTable', function () {
  var mountTable = function mountTable() {
    cy.realMount((0, _react2.jsx)(_.EuiInMemoryTable, {
      "data-test-subj": "cy-in-memory-table",
      tableCaption: "Demo of EuiInMemoryTable",
      items: users,
      columns: columns,
      pagination: true,
      sorting: {
        sort: {
          field: 'dateOfBirth',
          direction: 'desc'
        }
      }
    }));
    cy.get('div[data-test-subj="cy-in-memory-table"]').should('exist');
  };
  describe('Automated accessibility check', function () {
    describe('desktop', function () {
      beforeEach(function () {
        cy.viewport(1024, 768); // medium breakpoint
        mountTable();
      });
      it('has zero violations on first render', function () {
        cy.checkAxe();
      });
      it('has zero violations on pagination click', function () {
        cy.get('a[data-test-subj="pagination-button-1"]').realClick();
        cy.get('button[data-test-subj="pagination-button-1"]').should('be.disabled');
        cy.checkAxe();
      });
      it('has zero violations after number of rows is increased', function () {
        cy.get('button[data-test-subj="tablePaginationPopoverButton"]').realClick();
        cy.get('div[data-popover-open="true"]').should('exist');
        cy.get('button[data-test-subj="tablePagination-25-rows"]').realClick();
        cy.get('table.euiTable').find('tr.euiTableRow').should('have.length', 20);
        cy.checkAxe();
      });
      it('has zero violations after sorting on a column', function () {
        cy.get('button[data-test-subj="tableHeaderSortButton"]').first().focus();
        cy.realPress('{enter}');
        cy.checkAxe();
      });
      it('has zero violations when number of rows is increased by keyboard', function () {
        cy.get('button[data-test-subj="tablePaginationPopoverButton"]').focus().realPress('{enter}');
        cy.get('div[data-popover-open="true"]', {
          timeout: 1000
        }).should('exist');
        cy.repeatRealPress('Tab'); // Switched to Tab from ArrowDown because of flaky test runs
        cy.get('button[data-test-subj="tablePagination-25-rows"]', {
          timeout: 1000
        }).realPress('{enter}');
        cy.get('table.euiTable', {
          timeout: 1000
        }).find('tr.euiTableRow').should('have.length', 20);
        cy.checkAxe();
      });
      it('has zero violations when pagination is pressed', function () {
        cy.get('a[data-test-subj="pagination-button-1"]').focus().realPress('{enter}');
        cy.get('button[data-test-subj="pagination-button-1"]').should('be.disabled');
        cy.checkAxe();
      });
    });
    describe('mobile', function () {
      beforeEach(function () {
        cy.viewport(375, 667); // small breakpoint
        mountTable();
      });
      it('has zero violations on render', function () {
        cy.checkAxe();
      });
    });
  });
});