"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.saveWithConfirmation = saveWithConfirmation;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _constants = require("./constants");
var _confirm_modal_promise = require("./confirm_modal_promise");
var _content_management = require("../../content_management");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Attempts to create the current object using the serialized source. If an object already
 * exists, a warning message requests an overwrite confirmation.
 * @param source - serialized version of this object what will be indexed into elasticsearch.
 * @param savedObject - VisSavedObject
 * @param options - options to pass to the saved object create method
 * @param services - provides Kibana services savedObjectsClient and overlays
 * @returns {Promise} - A promise that is resolved with the objects id if the object is
 * successfully indexed. If the overwrite confirmation was rejected, an error is thrown with
 * a confirmRejected = true parameter so that case can be handled differently than
 * a create or index error.
 * @resolved {SimpleSavedObject}
 */
async function saveWithConfirmation(source, savedObject, options, services) {
  try {
    return await _content_management.visualizationsClient.create({
      data: source,
      options
    });
  } catch (err) {
    // record exists, confirm overwriting
    if ((0, _lodash.get)(err, 'res.status') === 409) {
      const confirmMessage = _i18n.i18n.translate('visualizations.confirmModal.overwriteConfirmationMessage', {
        defaultMessage: 'Are you sure you want to overwrite {title}?',
        values: {
          title: savedObject.title
        }
      });
      const title = _i18n.i18n.translate('visualizations.confirmModal.overwriteTitle', {
        defaultMessage: 'Overwrite {name}?',
        values: {
          name: savedObject.displayName
        }
      });
      const confirmButtonText = _i18n.i18n.translate('visualizations.confirmModal.overwriteButtonLabel', {
        defaultMessage: 'Overwrite'
      });
      return (0, _confirm_modal_promise.confirmModalPromise)(confirmMessage, title, confirmButtonText, services).then(() => _content_management.visualizationsClient.create({
        data: source,
        options: {
          overwrite: true,
          ...options
        }
      })).catch(() => Promise.reject(new Error(_constants.OVERWRITE_REJECTED)));
    }
    return await Promise.reject(err);
  }
}