"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.serializeState = exports.deserializeState = exports.deserializeSavedVisState = exports.deserializeSavedObjectState = void 0;
var _public = require("@kbn/data-plugin/public");
var _lodash = require("lodash");
var _services = require("../services");
var _saved_visualization_references = require("../utils/saved_visualization_references");
var _saved_visualize_utils = require("../utils/saved_visualize_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const deserializeState = async state => {
  var _state$references;
  if (!state.rawState) return {
    serializedVis: {
      data: {}
    }
  };
  let serializedState = (0, _lodash.cloneDeep)(state.rawState);
  if (serializedState.savedObjectId) {
    serializedState = await deserializeSavedObjectState(serializedState);
  } else if (serializedState.serializedVis) {
    // TODO remove once embeddable only exposes SerializedState
    // Canvas passes incoming embeddable state in getSerializedStateForChild
    // without this early return, serializedVis gets replaced in deserializeSavedVisState
    // and breaks adding a new by-value embeddable in Canvas
    return serializedState;
  }
  const references = (_state$references = state.references) !== null && _state$references !== void 0 ? _state$references : [];
  const deserializedSavedVis = deserializeSavedVisState(serializedState, references);
  return {
    ...serializedState,
    serializedVis: deserializedSavedVis
  };
};
exports.deserializeState = deserializeState;
const deserializeSavedVisState = (serializedState, references) => {
  var _serializedState$save, _deserializedReferenc, _deserializedReferenc2;
  const {
    data
  } = (_serializedState$save = serializedState.savedVis) !== null && _serializedState$save !== void 0 ? _serializedState$save : {
    data: {}
  };
  let serializedSearchSource = data.searchSource;
  let serializedReferences = [...references];
  if (data.searchSource && !('indexRefName' in data.searchSource)) {
    // due to a bug in 8.0, some visualizations were saved with an injected state - re-extract in that case and inject the upstream references because they might have changed
    const [extractedSearchSource, extractedReferences] = (0, _public.extractSearchSourceReferences)(serializedSearchSource);
    serializedSearchSource = extractedSearchSource;
    serializedReferences = [...references, ...extractedReferences];
  }
  const {
    references: deserializedReferences,
    deserializedSearchSource
  } = (0, _saved_visualization_references.deserializeReferences)({
    ...serializedState,
    savedVis: {
      ...serializedState.savedVis,
      data: {
        ...data,
        searchSource: serializedSearchSource
      }
    }
  }, serializedReferences);
  return {
    ...serializedState.savedVis,
    data: {
      ...data,
      searchSource: deserializedSearchSource,
      savedSearchId: (_deserializedReferenc = (_deserializedReferenc2 = deserializedReferences.find(r => r.name === 'search_0')) === null || _deserializedReferenc2 === void 0 ? void 0 : _deserializedReferenc2.id) !== null && _deserializedReferenc !== void 0 ? _deserializedReferenc : data.savedSearchId
    }
  };
};
exports.deserializeSavedVisState = deserializeSavedVisState;
const deserializeSavedObjectState = async ({
  savedObjectId,
  enhancements,
  uiState,
  timeRange,
  title: embeddableTitle,
  description: embeddableDescription,
  hidePanelTitles
}) => {
  // Load a saved visualization from the library
  const {
    title,
    description,
    visState,
    searchSource,
    searchSourceFields,
    savedSearchId,
    savedSearchRefName,
    uiStateJSON,
    ...savedObjectProperties
  } = await (0, _saved_visualize_utils.getSavedVisualization)({
    dataViews: (0, _services.getDataViews)(),
    search: (0, _services.getSearch)(),
    savedObjectsTagging: (0, _services.getSavedObjectTagging)().getTaggingApi(),
    spaces: (0, _services.getSpaces)(),
    i18n: (0, _services.getI18n)(),
    overlays: (0, _services.getOverlays)(),
    analytics: (0, _services.getAnalytics)(),
    theme: (0, _services.getTheme)(),
    userProfile: (0, _services.getUserProfile)()
  }, savedObjectId);
  const panelTitle = embeddableTitle !== null && embeddableTitle !== void 0 ? embeddableTitle : title;
  const panelDescription = embeddableDescription !== null && embeddableDescription !== void 0 ? embeddableDescription : description;
  return {
    savedVis: {
      title,
      type: visState.type,
      params: visState.params,
      uiState: uiState !== null && uiState !== void 0 ? uiState : uiStateJSON ? JSON.parse(uiStateJSON) : {},
      data: {
        aggs: visState.aggs,
        searchSource: searchSource !== null && searchSource !== void 0 ? searchSource : searchSourceFields,
        savedSearchId
      }
    },
    title: panelTitle,
    description: panelDescription,
    hidePanelTitles,
    savedObjectId,
    savedObjectProperties,
    linkedToLibrary: true,
    ...(timeRange ? {
      timeRange
    } : {}),
    ...(enhancements ? {
      enhancements
    } : {})
  };
};
exports.deserializeSavedObjectState = deserializeSavedObjectState;
const serializeState = ({
  serializedVis,
  // Serialize the vis before passing it to this function for easier testing
  titles,
  id,
  savedObjectProperties,
  linkedToLibrary,
  serializeDynamicActions,
  timeRange
}) => {
  var _serializeDynamicActi, _references$find;
  const {
    references,
    serializedSearchSource
  } = (0, _saved_visualization_references.serializeReferences)(serializedVis);
  const {
    rawState: dynamicActionsState,
    references: dynamicActionsReferences
  } = (_serializeDynamicActi = serializeDynamicActions === null || serializeDynamicActions === void 0 ? void 0 : serializeDynamicActions()) !== null && _serializeDynamicActi !== void 0 ? _serializeDynamicActi : {};

  // Serialize ONLY the savedObjectId. This ensures that when this vis is loaded again, it will always fetch the
  // latest revision of the saved object
  if (linkedToLibrary) {
    return {
      rawState: {
        ...(titles ? titles : {}),
        savedObjectId: id,
        ...dynamicActionsState,
        ...(!(0, _lodash.isEmpty)(serializedVis.uiState) ? {
          uiState: serializedVis.uiState
        } : {}),
        ...(timeRange ? {
          timeRange
        } : {})
      },
      references: [...references, ...(dynamicActionsReferences !== null && dynamicActionsReferences !== void 0 ? dynamicActionsReferences : [])]
    };
  }
  const savedSearchRefName = serializedVis.data.savedSearchId ? (_references$find = references.find(r => r.id === serializedVis.data.savedSearchId)) === null || _references$find === void 0 ? void 0 : _references$find.name : undefined;
  return {
    rawState: {
      ...(titles ? titles : {}),
      ...savedObjectProperties,
      ...dynamicActionsState,
      ...(timeRange ? {
        timeRange
      } : {}),
      savedVis: {
        ...serializedVis,
        id,
        data: {
          ...(0, _lodash.omit)(serializedVis.data, 'savedSearchId'),
          searchSource: serializedSearchSource,
          ...(savedSearchRefName ? {
            savedSearchRefName
          } : {})
        }
      }
    },
    references: [...references, ...(dynamicActionsReferences !== null && dynamicActionsReferences !== void 0 ? dynamicActionsReferences : [])]
  };
};
exports.serializeState = serializeState;