"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Dispatch = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _d = _interopRequireDefault(require("d3"));
var _lodash = require("lodash");
var _jquery = _interopRequireDefault(require("jquery"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Handles event responses
 *
 * @class Dispatch
 * @constructor
 * @param handler {Object} Reference to Handler Class Object
 */
class Dispatch {
  constructor(handler, uiSettings) {
    /**
     * Emit an event and all arguments to all listeners for an event name
     *
     * @param  {string} name
     * @param  {*} [arg...] - any number of arguments that will be applied to each handler
     * @return {Dispatch} - this, for chaining
     */
    (0, _defineProperty2.default)(this, "emit", (0, _lodash.rest)(function (name, args) {
      if (!this._listeners[name]) {
        return this;
      }
      const listeners = this.listeners(name);
      let i = -1;
      while (++i < listeners.length) {
        listeners[i].apply(this, args);
      }
      return this;
    }));
    this.handler = handler;
    this.uiSettings = uiSettings;
    this._listeners = {};
  }

  /**
   * Add an event handler
   *
   * @param  {string} name
   * @param  {function} handler
   * @return {Dispatch} - this, for chaining
   */
  on(name, handler) {
    let handlers = this._listeners[name];
    if (!handlers) {
      this._listeners[name] = [];
      handlers = this._listeners[name];
    }
    handlers.push(handler);
    return this;
  }

  /**
   * Remove an event handler
   *
   * @param  {string} name
   * @param  {function} [handler] - optional handler to remove, if no handler is
   *                              passed then all are removed
   * @return {Dispatch} - this, for chaining
   */
  off(name, handler) {
    if (!this._listeners[name]) {
      return this;
    }

    // remove a specific handler
    if (handler) {
      (0, _lodash.pull)(this._listeners[name], handler);
    }
    // or remove all listeners
    else {
      this._listeners[name] = null;
    }
    return this;
  }

  /**
   * Remove all event listeners bound to this emitter.
   *
   * @return {Dispatch} - this, for chaining
   */
  removeAllListeners() {
    this._listeners = {};
    return this;
  }
  /**
   * Get a list of the handler functions for a specific event
   *
   * @param  {string} name
   * @return {array[function]}
   */
  listeners(name) {
    return this._listeners[name] ? this._listeners[name].slice(0) : [];
  }

  /**
   * Get the count of handlers for a specific event
   *
   * @param  {string} [name] - optional event name to filter by
   * @return {number}
   */
  listenerCount(name) {
    if (name) {
      return (0, _lodash.size)(this._listeners[name]);
    }
    return (0, _lodash.reduce)(this._listeners, (count, handlers) => count + (0, _lodash.size)(handlers), 0);
  }
  _seriesClickResponse(data) {
    const points = [];
    ['xRaw', 'yRaw', 'zRaw', 'seriesRaw', 'rawData', 'tableRaw'].forEach(val => {
      if (data[val] && data[val].column !== undefined && data[val].row !== undefined) {
        points.push(data[val]);
      }
    });
    return points;
  }

  /**
   * Response to click  events
   *
   * @param d {Object} Data point
   * @returns event with list of data points related to the click
   */
  clickEventResponse(d, props = {}) {
    let isSlices = props.isSlices;
    if (isSlices === undefined) {
      const _data = _d.default.event.target.nearestViewportElement ? _d.default.event.target.nearestViewportElement.__data__ : _d.default.event.target.__data__;
      isSlices = !!(_data && _data.slices);
    }
    const data = d.input || d;
    return {
      data: this._seriesClickResponse(data)
    };
  }

  /**
   * Determine whether rendering a series is configured in percentage mode
   * Used to display a value percentage formatted in it's popover
   *
   * @param rawId {string} The rawId of series to check
   * @param series {Array} Array of all series data
   * @param visConfig {VisConfig}
   * @returns {Boolean}
   */
  _isSeriesInPercentageMode(rawId, series, visConfig) {
    if (!rawId || !Array.isArray(series) || !visConfig) {
      return false;
    }
    //find the primary id by the rawId, that id is used in the config's seriesParams
    const {
      id
    } = series.find(series => series.rawId === rawId);
    if (!id) {
      return false;
    }

    //find the matching seriesParams of the series, to get the id of the valueAxis
    const seriesParams = visConfig.get('seriesParams', []);
    const {
      valueAxis: valueAxisId
    } = seriesParams.find(param => param.data.id === id) || {};
    if (!valueAxisId) {
      return false;
    }
    const usedValueAxis = visConfig.get('valueAxes', []).find(valueAxis => valueAxis.id === valueAxisId);
    return (0, _lodash.get)(usedValueAxis, 'scale.mode') === 'percentage';
  }

  /**
   * Response to hover events
   *
   * @param d {Object} Data point
   * @param i {Number} Index number of data point
   * @returns {{value: *, point: *, label: *, color: *, pointIndex: *,
   * series: *, config: *, data: (Object|*),
   * e: (d3.event|*), handler: (Object|*)}} Event response object
   */
  eventResponse(d, i) {
    const datum = d._input || d;
    const data = _d.default.event.target.nearestViewportElement ? _d.default.event.target.nearestViewportElement.__data__ : _d.default.event.target.__data__;
    const label = d.label ? d.label : d.series || 'Count';
    const isSeries = !!(data && data.series);
    const isSlices = !!(data && data.slices);
    const series = isSeries ? data.series : undefined;
    const slices = isSlices ? data.slices : undefined;
    const handler = this.handler;
    const color = (0, _lodash.get)(handler, 'data.color');
    const config = handler && handler.visConfig;
    const isPercentageMode = this._isSeriesInPercentageMode(d.seriesId, series, config);
    const eventData = {
      value: d.y,
      point: datum,
      datum,
      label,
      color: color ? color(label) : undefined,
      pointIndex: i,
      series,
      slices,
      config,
      data,
      e: _d.default.event,
      handler,
      isPercentageMode
    };
    return eventData;
  }

  /**
   * Returns a function that adds events and listeners to a D3 selection
   *
   * @method addEvent
   * @param event {String}
   * @param callback {Function}
   * @returns {Function}
   */
  addEvent(event, callback) {
    return function (selection) {
      selection.each(function () {
        const element = _d.default.select(this);
        if (typeof callback === 'function') {
          return element.on(event, callback);
        }
      });
    };
  }

  /**
   *
   * @method addHoverEvent
   * @returns {Function}
   */
  addHoverEvent() {
    const self = this;
    const isClickable = this.listenerCount('click') > 0;
    const addEvent = this.addEvent;
    const $el = this.handler.el;
    if (!this.handler.highlight) {
      this.handler.highlight = self.getHighlighter();
    }
    function hover(d, i) {
      // Add pointer if item is clickable
      if (isClickable) {
        self.addMousePointer.call(this, arguments);
      }
      self.handler.highlight.call(this, $el);
      self.emit('hover', self.eventResponse(d, i));
    }
    return addEvent('mouseover', hover);
  }

  /**
   *
   * @method addMouseoutEvent
   * @returns {Function}
   */
  addMouseoutEvent() {
    const self = this;
    const addEvent = this.addEvent;
    const $el = this.handler.el;
    if (!this.handler.unHighlight) {
      this.handler.unHighlight = self.unHighlight;
    }
    function mouseout() {
      self.handler.unHighlight.call(this, $el);
    }
    return addEvent('mouseout', mouseout);
  }

  /**
   *
   * @method addClickEvent
   * @returns {Function}
   */
  addClickEvent() {
    const onClick = d => this.emit('click', this.clickEventResponse(d));
    return this.addEvent('click', onClick);
  }

  /**
   * Determine if we will allow brushing
   *
   * @method allowBrushing
   * @returns {Boolean}
   */
  allowBrushing() {
    const xAxis = this.handler.categoryAxes[0];

    //Allow brushing for ordered axis - date histogram and histogram
    return Boolean(xAxis.ordered);
  }

  /**
   * Determine if brushing is currently enabled
   *
   * @method isBrushable
   * @returns {Boolean}
   */
  isBrushable() {
    return this.allowBrushing() && this.listenerCount('brush') > 0;
  }

  /**
   *
   * @param svg
   * @returns {Function}
   */
  addBrushEvent(svg) {
    if (!this.isBrushable()) return;
    const xScale = this.handler.categoryAxes[0].getScale();
    this.createBrush(xScale, svg);
  }

  /**
   * Mouseover Behavior
   *
   * @method addMousePointer
   * @returns {d3.Selection}
   */
  addMousePointer() {
    return _d.default.select(this).style('cursor', 'pointer');
  }

  /**
   * return function to Highlight the element that is under the cursor
   * by reducing the opacity of all the elements on the graph.
   * @method getHighlighter
   */
  getHighlighter() {
    return function highlight(element) {
      const label = this.getAttribute('data-label');
      if (!label) return;
      (0, _jquery.default)(element).parent().find('[data-label]').css('opacity', 1) //Opacity 1 is needed to avoid the css application
      .not((els, el) => String((0, _jquery.default)(el).data('label')) === label).css('opacity', 0.5);
    };
  }

  /**
   * Mouseout Behavior
   *
   * @param element {d3.Selection}
   * @method unHighlight
   */
  unHighlight(element) {
    (0, _jquery.default)('[data-label]', element.parentNode).css('opacity', 1);
  }

  /**
   * Adds D3 brush to SVG and returns the brush function
   *
   * @param xScale {Function} D3 xScale function
   * @param svg {HTMLElement} Reference to SVG
   * @returns {*} Returns a D3 brush function and a SVG with a brush group attached
   */
  createBrush(xScale, svg) {
    const self = this;
    const {
      width,
      height
    } = svg.node().getBBox();
    const isHorizontal = self.handler.categoryAxes[0].axisConfig.isHorizontal();

    // Brush scale
    const brush = _d.default.svg.brush();
    if (isHorizontal) {
      brush.x(xScale);
    } else {
      brush.y(xScale);
    }
    brush.on('brushend', function brushEnd() {
      // Assumes data is selected at the chart level
      // In this case, the number of data objects should always be 1
      const data = _d.default.select(this).data()[0];
      const isTimeSeries = data.ordered && data.ordered.date;

      // Allows for brushing on d3.scale.ordinal()
      const selected = xScale.domain().filter(d => brush.extent()[0] <= xScale(d) && xScale(d) <= brush.extent()[1]);
      const range = isTimeSeries ? brush.extent() : selected;
      return self.emit('brush', {
        range,
        data
      });
    });

    // if `addBrushing` is true, add brush canvas
    if (self.listenerCount('brush')) {
      const rect = svg.insert('g', 'g').attr('class', 'brush').call(brush).call(brushG => {
        // hijack the brush start event to filter out right/middle clicks
        const brushHandler = brushG.on('mousedown.brush');
        if (!brushHandler) return; // touch events in use
        brushG.on('mousedown.brush', function () {
          if (validBrushClick(_d.default.event)) brushHandler.apply(this, arguments);
        });
      }).selectAll('rect');
      if (isHorizontal) {
        rect.attr('height', height);
      } else {
        rect.attr('width', width);
      }
      return brush;
    }
  }
}
exports.Dispatch = Dispatch;
function validBrushClick(event) {
  return event.button === 0;
}