"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkObjectForFunctionProperty = checkObjectForFunctionProperty;
exports.normalizeDate = normalizeDate;
exports.normalizeObject = normalizeObject;
exports.normalizeString = normalizeString;
var _std = require("@kbn/std");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function normalizeDate(date) {
  if (typeof date === 'number') {
    return !isNaN(date) ? date : null;
  } else if (date instanceof Date) {
    return date;
  } else {
    return normalizeObject(date);
  }
}

/*
Recursive function to check a nested object for a function property
This function should run before JSON.stringify to ensure that functions such as toJSON
are not invoked. We dont use the replacer function as it doesnt catch the toJSON function
*/
function checkObjectForFunctionProperty(object) {
  if (object === null || object === undefined) {
    return false;
  }
  if (typeof object === 'function') {
    return true;
  }
  if (object && typeof object === 'object') {
    return Object.values(object).some(value => typeof value === 'function' || checkObjectForFunctionProperty(value));
  }
  return false;
}
/*
  We want to be strict here when an object is passed to a Vega function
  - NaN (will be converted to null)
  - undefined (key will be removed)
  - Date (will be replaced by its toString value)
  - will throw an error when a function is found
  */
function normalizeObject(object) {
  if (checkObjectForFunctionProperty(object)) {
    throw new Error('a function cannot be used as a property name');
  }
  const normalizedObject = object ? JSON.parse(JSON.stringify(object)) : null;
  (0, _std.ensureNoUnsafeProperties)(normalizedObject);
  return normalizedObject;
}
function normalizeString(string) {
  return typeof string === 'string' ? string : undefined;
}