"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.rollUsageCountersIndices = rollUsageCountersIndices;
var _moment = _interopRequireDefault(require("moment"));
var _lodash = require("lodash");
var _constants = require("./constants");
var _2 = require("..");
var _kuery_utils = require("../common/kuery_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// Process 1000 at a time as a compromise of speed and overload
const ROLLUP_BATCH_SIZE = 1000;
async function rollUsageCountersIndices({
  logger,
  getRegisteredUsageCounters,
  internalRepository,
  now = (0, _moment.default)()
}) {
  if (!internalRepository) {
    return;
  }
  let cleanupCounter = 0;
  try {
    const counterQueue = getRegisteredUsageCounters();
    while (counterQueue.length > 0) {
      var _counter$retentionPer;
      const counter = counterQueue.shift();
      const findParams = {
        type: _2.USAGE_COUNTERS_SAVED_OBJECT_TYPE,
        filter: (0, _kuery_utils.usageCountersSearchParamsToKueryFilter)({
          domainId: counter.domainId,
          to: (0, _moment.default)(now)
          // get documents that are OLDER than the retention period
          .subtract(1 + ((_counter$retentionPer = counter.retentionPeriodDays) !== null && _counter$retentionPer !== void 0 ? _counter$retentionPer : _constants.USAGE_COUNTERS_KEEP_DOCS_FOR_DAYS), 'days').toISOString()
        }),
        sortField: 'updated_at',
        sortOrder: 'asc',
        namespaces: ['*'],
        perPage: ROLLUP_BATCH_SIZE
      };
      const {
        saved_objects: rawUiCounterDocs
      } = await internalRepository.find(findParams);
      if (rawUiCounterDocs.length) {
        const toDelete = rawUiCounterDocs.map(({
          id,
          type,
          namespaces
        }) => {
          var _namespaces$;
          return {
            id,
            type,
            namespace: (_namespaces$ = namespaces === null || namespaces === void 0 ? void 0 : namespaces[0]) !== null && _namespaces$ !== void 0 ? _namespaces$ : 'default'
          };
        });
        cleanupCounter += toDelete.length;
        logger.debug(`[Rollups] Cleaning ${toDelete.length} old Usage Counters saved objects under domain '${counter.domainId}'`);
        const toDeleteByNamespace = (0, _lodash.groupBy)(toDelete, 'namespace');

        // perform a Bulk delete for each namespace
        await Promise.all(Object.entries(toDeleteByNamespace).map(([namespace, counters]) => internalRepository.bulkDelete(counters.map(({
          namespace: _,
          ...props
        }) => ({
          ...props
        })), {
          namespace
        })));
        if (toDelete.length === ROLLUP_BATCH_SIZE) {
          // we found a lot of old Usage Counters, put the counter back in the queue, as there might be more
          counterQueue.push(counter);
        }
      }
    }
  } catch (err) {
    logger.warn(`Failed to rollup Usage Counters saved objects.`);
    logger.warn(err);
  }
  if (cleanupCounter) {
    logger.debug(`[Rollups] Cleaned ${cleanupCounter} Usage Counters saved objects`);
  }
  return cleanupCounter;
}