"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDeleteIndexRoute = registerDeleteIndexRoute;
var _configSchema = require("@kbn/config-schema");
var _elasticsearch = require("@elastic/elasticsearch");
var _constants = require("../../common/constants");
var _es_version_precheck = require("../lib/es_version_precheck");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerDeleteIndexRoute({
  router,
  lib: {
    handleEsError
  }
}) {
  router.delete({
    path: `${_constants.API_BASE_PATH}/delete_index/{index}`,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es and saved object clients for authorization'
      }
    },
    options: {
      access: 'public',
      summary: `Remove an index from the cluster. This operation is irreversible and should be used with caution.`
    },
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      })
    }
  }, (0, _es_version_precheck.versionCheckHandlerWrapper)(async ({
    core
  }, request, response) => {
    const {
      elasticsearch: {
        client
      }
    } = await core;
    const {
      index
    } = request.params;
    try {
      await client.asCurrentUser.indices.delete({
        index
      });
      return response.ok();
    } catch (err) {
      if (err instanceof _elasticsearch.errors.ResponseError) {
        return handleEsError({
          error: err,
          response
        });
      }
      throw err;
    }
  }));
}