"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.search = void 0;
var _common = require("@kbn/data-plugin/common");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Handles optional access to RequestAdapter
 */
const requestInspector = (inspectorAdapter, {
  requestName
}) => {
  const requestId = `${Date.now()}`;
  const inspectorRequest = inspectorAdapter === null || inspectorAdapter === void 0 ? void 0 : inspectorAdapter.start(requestName, {
    id: requestId
  });
  return {
    recordRequestError(error) {
      if (!inspectorRequest) {
        return;
      }
      inspectorRequest.error({
        json: error
      });
    },
    recordRequestCompletion(searchRequest, response) {
      var _searchRequest$params;
      if (!inspectorRequest) {
        return;
      }
      inspectorRequest.stats({}).ok({
        json: response
      });
      inspectorRequest.json(((_searchRequest$params = searchRequest.params) === null || _searchRequest$params === void 0 ? void 0 : _searchRequest$params.body) || {});
    },
    resetRequest() {
      if (!inspectorAdapter) {
        return;
      }
      inspectorAdapter.resetRequest(requestId);
    }
  };
};

/**
 * This is a searchService wrapper that will instrument your query with `inspector` and turn it into a Promise,
 * resolved when complete result set is returned or rejected on any error, other than Abort.
 */
const search = async (searchService, searchRequest, {
  inspectorAdapter,
  requestName,
  signal,
  skipDefaultStrategy = false
}) => {
  const inspect = requestInspector(inspectorAdapter, {
    requestName
  });
  return new Promise((resolve, reject) => {
    searchService.search(searchRequest, {
      abortSignal: signal,
      ...(skipDefaultStrategy ? {} : {
        strategy: _constants.THREAT_INTELLIGENCE_SEARCH_STRATEGY_NAME
      })
    }).subscribe({
      next: response => {
        if (!(0, _common.isRunningResponse)(response)) {
          inspect.recordRequestCompletion(searchRequest, response);
          resolve(response.rawResponse);
        }
      },
      error: requestError => {
        if (requestError instanceof Error && requestError.name.includes('Abort')) {
          inspect.resetRequest();
        } else {
          inspect.recordRequestError(requestError);
        }
        searchService.showError(requestError);
        reject(requestError);
      }
    });
  });
};
exports.search = search;