"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TaskRunningStage = exports.TaskRunResult = exports.TaskManagerRunner = exports.TASK_MANAGER_TRANSACTION_TYPE_MARK_AS_RUNNING = exports.TASK_MANAGER_TRANSACTION_TYPE = exports.TASK_MANAGER_RUN_TRANSACTION_TYPE = exports.EMPTY_RUN_RESULT = void 0;
exports.asPending = asPending;
exports.asRan = asRan;
exports.asReadyToRun = asReadyToRun;
exports.getTaskDelayInSeconds = getTaskDelayInSeconds;
exports.isPending = isPending;
exports.isReadyToRun = isReadyToRun;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _uuid = require("uuid");
var _apmUtils = require("@kbn/apm-utils");
var _lodash = require("lodash");
var _server = require("@kbn/core/server");
var _spacesUtils = require("@kbn/spaces-utils");
var _coreHttpServerUtils = require("@kbn/core-http-server-utils");
var _result_type = require("../lib/result_type");
var _task_events = require("../task_events");
var _intervals = require("../lib/intervals");
var _wrapped_logger = require("../lib/wrapped_logger");
var _task = require("../task");
var _errors = require("./errors");
var _config = require("../config");
var _task_validator = require("../task_validator");
var _get_retry_at = require("../lib/get_retry_at");
var _get_next_run_at = require("../lib/get_next_run_at");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * This module contains the core logic for running an individual task.
 * It handles the full lifecycle of a task run, including error handling,
 * rescheduling, middleware application, etc.
 */

const EMPTY_RUN_RESULT = exports.EMPTY_RUN_RESULT = {
  state: {}
};
const TASK_MANAGER_RUN_TRANSACTION_TYPE = exports.TASK_MANAGER_RUN_TRANSACTION_TYPE = 'task-run';
const TASK_MANAGER_TRANSACTION_TYPE = exports.TASK_MANAGER_TRANSACTION_TYPE = 'task-manager';
const TASK_MANAGER_TRANSACTION_TYPE_MARK_AS_RUNNING = exports.TASK_MANAGER_TRANSACTION_TYPE_MARK_AS_RUNNING = 'mark-task-as-running';
let TaskRunningStage = exports.TaskRunningStage = /*#__PURE__*/function (TaskRunningStage) {
  TaskRunningStage["PENDING"] = "PENDING";
  TaskRunningStage["READY_TO_RUN"] = "READY_TO_RUN";
  TaskRunningStage["RAN"] = "RAN";
  return TaskRunningStage;
}({});
let TaskRunResult = exports.TaskRunResult = /*#__PURE__*/function (TaskRunResult) {
  TaskRunResult["Success"] = "Success";
  TaskRunResult["SuccessRescheduled"] = "Success";
  TaskRunResult["RetryScheduled"] = "RetryScheduled";
  TaskRunResult["Failed"] = "Failed";
  TaskRunResult["Deleted"] = "Deleted";
  return TaskRunResult;
}({}); // A ConcreteTaskInstance which we *know* has a `startedAt` Date on it
// The three possible stages for a Task Runner - Pending -> ReadyToRun -> Ran
/**
 * Runs a background task, ensures that errors are properly handled,
 * allows for cancellation.
 *
 * @export
 * @class TaskManagerRunner
 * @implements {TaskRunner}
 */
class TaskManagerRunner {
  /**
   * Creates an instance of TaskManagerRunner.
   * @param {Opts} opts
   * @prop {Logger} logger - The task manager logger
   * @prop {TaskDefinition} definition - The definition of the task being run
   * @prop {ConcreteTaskInstance} instance - The record describing this particular task instance
   * @prop {Updatable} store - The store used to read / write tasks instance info
   * @prop {BeforeRunFunction} beforeRun - A function that adjusts the run context prior to running the task
   * @memberof TaskManagerRunner
   */
  constructor({
    basePathService,
    instance,
    definitions,
    logger,
    store,
    beforeRun,
    beforeMarkRunning,
    defaultMaxAttempts,
    onTaskEvent = _lodash.identity,
    executionContext,
    usageCounter,
    config,
    allowReadingInvalidState,
    strategy,
    getPollInterval
  }) {
    (0, _defineProperty2.default)(this, "task", void 0);
    (0, _defineProperty2.default)(this, "instance", void 0);
    (0, _defineProperty2.default)(this, "definitions", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "bufferedTaskStore", void 0);
    (0, _defineProperty2.default)(this, "beforeRun", void 0);
    (0, _defineProperty2.default)(this, "beforeMarkRunning", void 0);
    (0, _defineProperty2.default)(this, "onTaskEvent", void 0);
    (0, _defineProperty2.default)(this, "defaultMaxAttempts", void 0);
    (0, _defineProperty2.default)(this, "uuid", void 0);
    (0, _defineProperty2.default)(this, "basePathService", void 0);
    (0, _defineProperty2.default)(this, "executionContext", void 0);
    (0, _defineProperty2.default)(this, "usageCounter", void 0);
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "taskValidator", void 0);
    (0, _defineProperty2.default)(this, "claimStrategy", void 0);
    (0, _defineProperty2.default)(this, "getPollInterval", void 0);
    (0, _defineProperty2.default)(this, "rescheduleFailedRun", failureResult => {
      const {
        state,
        error
      } = failureResult;
      const {
        schedule,
        attempts
      } = this.instance.task;
      if (this.shouldTryToScheduleRetry() && !(0, _errors.isUnrecoverableError)(error)) {
        // if we're retrying, keep the number of attempts

        let reschedule = {};
        if (failureResult.runAt) {
          reschedule = {
            runAt: failureResult.runAt
          };
        } else if (failureResult.schedule) {
          reschedule = {
            schedule: failureResult.schedule
          };
        } else if (schedule) {
          reschedule = {
            schedule
          };
        } else {
          // when result.error is truthy, then we're retrying because it failed
          reschedule = {
            runAt: (0, _get_retry_at.getRetryDate)({
              attempts,
              error
            })
          };
        }
        if (reschedule.runAt || reschedule.schedule) {
          return (0, _result_type.asOk)({
            state,
            attempts,
            ...reschedule
          });
        }
      }

      // scheduling a retry isn't possible,mark task as failed
      return (0, _result_type.asErr)({
        status: _task.TaskStatus.Failed
      });
    });
    this.basePathService = basePathService;
    this.instance = asPending(sanitizeInstance(instance));
    this.definitions = definitions;
    this.logger = logger;
    this.bufferedTaskStore = store;
    this.beforeRun = beforeRun;
    this.beforeMarkRunning = beforeMarkRunning;
    this.onTaskEvent = onTaskEvent;
    this.defaultMaxAttempts = defaultMaxAttempts;
    this.executionContext = executionContext;
    this.usageCounter = usageCounter;
    this.uuid = (0, _uuid.v4)();
    this.config = config;
    this.taskValidator = new _task_validator.TaskValidator({
      logger: this.logger,
      definitions: this.definitions,
      allowReadingInvalidState
    });
    this.claimStrategy = strategy;
    this.getPollInterval = getPollInterval;
  }

  /**
   * Gets the id of this task instance.
   */
  get id() {
    return this.instance.task.id;
  }

  /**
   * Gets the execution id of this task instance.
   */
  get taskExecutionId() {
    return `${this.id}::${this.uuid}`;
  }

  /**
   * Test whether given execution ID identifies a different execution of this same task
   * @param id
   */
  isSameTask(executionId) {
    const executionIdParts = executionId.split('::');
    const executionIdCompare = executionIdParts.length > 0 ? executionIdParts[0] : executionId;
    return executionIdCompare === this.id;
  }

  /**
   * Gets the task type of this task instance.
   */
  get taskType() {
    return this.instance.task.taskType;
  }

  /**
   * Get the stage this TaskRunner is at
   */
  get stage() {
    return this.instance.stage;
  }

  /**
   * Gets the task defintion from the dictionary.
   */
  get definition() {
    return this.definitions.get(this.taskType);
  }

  /**
   * Gets the time at which this task will expire.
   */
  get expiration() {
    return (0, _intervals.intervalFromDate)(
    // if the task is running, use it's started at, otherwise use the timestamp at
    // which it was last updated
    // this allows us to catch tasks that remain in Pending/Finalizing without being
    // cleaned up
    isReadyToRun(this.instance) ? this.instance.task.startedAt : this.instance.timestamp, this.timeout);
  }

  /*
   * Gets the timeout of the current task. Uses the timeout
   * defined by the task type unless this is an ad-hoc task that specifies an override
   */
  get timeout() {
    return (0, _get_retry_at.getTimeout)(this.instance.task, this.definition);
  }

  /**
   * Gets the duration of the current task run
   */
  get startedAt() {
    return this.instance.task.startedAt;
  }

  /**
   * Gets whether or not this task has run longer than its expiration setting allows.
   */
  get isExpired() {
    return this.expiration < new Date();
  }

  /**
   * Returns true whenever the task is ad hoc and has ran out of attempts. When true before
   * running a task, the task should be deleted instead of ran.
   */
  get isAdHocTaskAndOutOfAttempts() {
    if (this.instance.task.status === 'running') {
      // This function gets called with tasks marked as running when using MGET, so attempts is already incremented
      return !this.instance.task.schedule && this.instance.task.attempts > this.getMaxAttempts();
    }
    return !this.instance.task.schedule && this.instance.task.attempts >= this.getMaxAttempts();
  }

  /**
   * Returns a log-friendly representation of this task.
   */
  toString() {
    return `${this.taskType} "${this.id}"`;
  }

  /**
   * Runs the task, handling the task result, errors, etc, rescheduling if need
   * be. NOTE: the time of applying the middleware's beforeRun is incorporated
   * into the total timeout time the task in configured with. We may decide to
   * start the timer after beforeRun resolves
   *
   * @returns {Promise<Result<SuccessfulRunResult, FailedRunResult>>}
   */
  async run() {
    const definition = this.definition;
    if (!definition) {
      throw new Error(`Running task ${this} failed because it has no definition`);
    }
    if (!isReadyToRun(this.instance)) {
      throw new Error(`Running task ${this} failed as it ${isPending(this.instance) ? `isn't ready to be ran` : `has already been ran`}`);
    }
    this.logger.debug(`Running task ${this}`, {
      tags: ['task:start', this.id, this.taskType]
    });
    const apmTrans = _elasticApmNode.default.startTransaction(this.taskType, TASK_MANAGER_RUN_TRANSACTION_TYPE, {
      childOf: this.instance.task.traceparent
    });
    const stopTaskTimer = (0, _task_events.startTaskTimerWithEventLoopMonitoring)(this.config.event_loop_delay);

    // Validate state
    const stateValidationResult = this.validateTaskState(this.instance.task);
    if (stateValidationResult.error) {
      const processedResult = await (0, _apmUtils.withSpan)({
        name: 'process result',
        type: 'task manager'
      }, () => this.processResult((0, _result_type.asErr)({
        error: stateValidationResult.error,
        state: stateValidationResult.taskInstance.state,
        shouldValidate: false
      }), stopTaskTimer()));
      if (apmTrans) apmTrans.end('failure');
      return processedResult;
    }
    const modifiedContext = await this.beforeRun({
      taskInstance: stateValidationResult.taskInstance
    });
    this.onTaskEvent((0, _task_events.asTaskManagerStatEvent)('runDelay', (0, _result_type.asOk)(getTaskDelayInSeconds(this.instance.task.scheduledAt))));
    try {
      var _modifiedContext$task;
      const sanitizedTaskInstance = (0, _lodash.omit)(modifiedContext.taskInstance, ['apiKey', 'userScope']);
      const fakeRequest = this.getFakeKibanaRequest(modifiedContext.taskInstance.apiKey, (_modifiedContext$task = modifiedContext.taskInstance.userScope) === null || _modifiedContext$task === void 0 ? void 0 : _modifiedContext$task.spaceId);
      this.task = definition.createTaskRunner({
        taskInstance: sanitizedTaskInstance,
        fakeRequest
      });
      const ctx = {
        type: 'task manager',
        name: `run ${this.instance.task.taskType}`,
        id: this.instance.task.id,
        description: 'run task'
      };
      const result = await this.executionContext.withContext(ctx, () => (0, _apmUtils.withSpan)({
        name: 'run',
        type: 'task manager'
      }, () => this.task.run()));
      const validatedResult = this.validateResult(result);
      const processedResult = await (0, _apmUtils.withSpan)({
        name: 'process result',
        type: 'task manager'
      }, () => this.processResult(validatedResult, stopTaskTimer()));
      if (apmTrans) apmTrans.end('success');
      return processedResult;
    } catch (err) {
      const errorSource = (0, _errors.isUserError)(err) ? _constants.TaskErrorSource.USER : _constants.TaskErrorSource.FRAMEWORK;
      this.logger.error(`Task ${this} failed: ${err}`, {
        tags: [this.taskType, this.instance.task.id, 'task-run-failed', `${errorSource}-error`],
        error: {
          stack_trace: err.stack
        }
      });
      // in error scenario, we can not get the RunResult
      // re-use modifiedContext's state, which is correct as of beforeRun
      const processedResult = await (0, _apmUtils.withSpan)({
        name: 'process result',
        type: 'task manager'
      }, () => this.processResult((0, _result_type.asErr)({
        error: err,
        state: modifiedContext.taskInstance.state
      }), stopTaskTimer()));
      if (apmTrans) apmTrans.end('failure');
      return processedResult;
    } finally {
      this.logger.debug(`Task ${this} ended`, {
        tags: ['task:end', this.id, this.taskType]
      });
    }
  }
  validateTaskState(taskInstance) {
    const {
      taskType,
      id
    } = taskInstance;
    try {
      const validatedTaskInstance = this.taskValidator.getValidatedTaskInstanceFromReading(taskInstance);
      return {
        taskInstance: validatedTaskInstance,
        error: null
      };
    } catch (error) {
      this.logger.warn(`Task (${taskType}/${id}) has a validation error: ${error.message}`);
      return {
        taskInstance,
        error
      };
    }
  }
  async removeTask() {
    var _this$task;
    await this.bufferedTaskStore.remove(this.id);
    if ((_this$task = this.task) !== null && _this$task !== void 0 && _this$task.cleanup) {
      try {
        await this.task.cleanup();
      } catch (e) {
        this.logger.error(`Error encountered when running onTaskRemoved() hook for ${this}: ${e.message}`);
      }
    }
  }

  /**
   * Attempts to claim exclusive rights to run the task. If the attempt fails
   * with a 409 (http conflict), we assume another Kibana instance beat us to the punch.
   *
   * @returns {Promise<boolean>}
   */
  async markTaskAsRunning() {
    if (!isPending(this.instance)) {
      throw new Error(`Marking task ${this} as running has failed as it ${isReadyToRun(this.instance) ? `is already running` : `has already been ran`}`);
    }

    // mget claim strategy sets the task to `running` during the claim cycle
    // so this update to mark the task as running is unnecessary
    if (this.claimStrategy === _config.CLAIM_STRATEGY_MGET) {
      this.instance = asReadyToRun(this.instance.task);
      return true;
    }
    const apmTrans = _elasticApmNode.default.startTransaction(TASK_MANAGER_TRANSACTION_TYPE_MARK_AS_RUNNING, TASK_MANAGER_TRANSACTION_TYPE);
    apmTrans.addLabels({
      entityId: this.taskType
    });
    const now = new Date();
    try {
      var _getRetryAt;
      const {
        taskInstance
      } = await this.beforeMarkRunning({
        taskInstance: this.instance.task
      });
      const attempts = taskInstance.attempts + 1;
      const ownershipClaimedUntil = taskInstance.retryAt;
      const {
        id
      } = taskInstance;
      const timeUntilClaimExpires = howManyMsUntilOwnershipClaimExpires(ownershipClaimedUntil);
      if (timeUntilClaimExpires < 0) {
        this.logger.debug(`[Task Runner] Task ${id} started after ownership expired (${Math.abs(timeUntilClaimExpires)}ms after expiry)`);
      }
      this.instance = asReadyToRun(await this.bufferedTaskStore.update({
        ...taskWithoutEnabled(taskInstance),
        status: _task.TaskStatus.Running,
        startedAt: now,
        attempts,
        retryAt: (_getRetryAt = (0, _get_retry_at.getRetryAt)(taskInstance, this.definition)) !== null && _getRetryAt !== void 0 ? _getRetryAt : null
        // This is a safe conversion as we're setting the startAt above
      }, {
        validate: false
      }));
      const timeUntilClaimExpiresAfterUpdate = howManyMsUntilOwnershipClaimExpires(ownershipClaimedUntil);
      if (timeUntilClaimExpiresAfterUpdate < 0) {
        this.logger.debug(`[Task Runner] Task ${id} ran after ownership expired (${Math.abs(timeUntilClaimExpiresAfterUpdate)}ms after expiry)`);
      }
      if (apmTrans) apmTrans.end('success');
      this.onTaskEvent((0, _task_events.asTaskMarkRunningEvent)(this.id, (0, _result_type.asOk)(this.instance.task)));
      return true;
    } catch (error) {
      if (apmTrans) apmTrans.end('failure');
      this.onTaskEvent((0, _task_events.asTaskMarkRunningEvent)(this.id, (0, _result_type.asErr)(error)));
      if (!_server.SavedObjectsErrorHelpers.isConflictError(error)) {
        if (!_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
          // try to release claim as an unknown failure prevented us from marking as running
          (0, _result_type.mapErr)(errReleaseClaim => {
            this.logger.error(`[Task Runner] Task ${this.id} failed to release claim after failure: Error: ${errReleaseClaim.message}`);
          }, await this.releaseClaimAndIncrementAttempts());
        }
        throw error;
      }
    }
    return false;
  }

  /**
   * Attempts to cancel the task.
   *
   * @returns {Promise<void>}
   */
  async cancel() {
    const {
      task
    } = this;
    if (task !== null && task !== void 0 && task.cancel) {
      // it will cause the task state of "running" to be cleared
      this.task = undefined;
      return task.cancel();
    }
    this.logger.debug(`The task ${this} is not cancellable.`);
  }
  validateResult(result) {
    return (0, _task.isFailedRunResult)(result) ? (0, _result_type.asErr)({
      ...result,
      error: result.error
    }) : (0, _result_type.asOk)({
      ...(result || EMPTY_RUN_RESULT)
    });
  }
  async releaseClaimAndIncrementAttempts() {
    return (0, _result_type.promiseResult)(this.bufferedTaskStore.partialUpdate({
      id: this.instance.task.id,
      version: this.instance.task.version,
      status: _task.TaskStatus.Idle,
      attempts: this.instance.task.attempts + 1,
      startedAt: null,
      retryAt: null,
      ownerId: null
    }, {
      validate: false,
      doc: this.instance.task
    }));
  }
  shouldTryToScheduleRetry() {
    if (this.instance.task.schedule) {
      return true;
    }
    if (this.isExpired) {
      this.logger.warn(`Skipping reschedule for task ${this} due to the task expiring`);
      return false;
    }
    return this.instance.task.attempts < this.getMaxAttempts();
  }
  shouldUpdateExpiredTask() {
    if (!this.instance.task.schedule || !this.instance.task.schedule.interval) {
      // if the task does not have a schedule interval, we will not update it on timeout
      return false;
    }
    const timeoutDuration = (0, _intervals.parseIntervalAsMillisecond)(this.timeout);
    const scheduleDuration = (0, _intervals.parseIntervalAsMillisecond)(this.instance.task.schedule.interval);
    return scheduleDuration > timeoutDuration;
  }
  async processResultForRecurringTask(result) {
    const hasTaskRunFailed = (0, _result_type.isOk)(result);
    let shouldTaskBeDisabled = false;
    const fieldUpdates = (0, _lodash.flow)(
    // if running the task has failed ,try to correct by scheduling a retry in the near future
    (0, _result_type.mapErr)(this.rescheduleFailedRun),
    // if retrying is possible (new runAt) or this is an recurring task - reschedule
    (0, _result_type.mapOk)(({
      runAt,
      schedule: reschedule,
      state,
      attempts = 0,
      shouldDeleteTask,
      shouldDisableTask
    }) => {
      if (shouldDeleteTask) {
        // set the status to failed so task will get deleted
        return (0, _result_type.asOk)({
          status: _task.TaskStatus.ShouldDelete
        });
      }
      if (shouldDisableTask) {
        shouldTaskBeDisabled = true;
        return (0, _result_type.asOk)({
          status: _task.TaskStatus.Idle
        });
      }
      const updatedTaskSchedule = reschedule !== null && reschedule !== void 0 ? reschedule : this.instance.task.schedule;
      return (0, _result_type.asOk)({
        runAt: runAt || (0, _get_next_run_at.getNextRunAt)({
          runAt: this.instance.task.runAt,
          startedAt: this.instance.task.startedAt,
          schedule: updatedTaskSchedule
        }, this.getPollInterval(), this.logger),
        state,
        schedule: updatedTaskSchedule,
        attempts,
        status: _task.TaskStatus.Idle
      });
    }), _result_type.unwrap)(result);
    if (fieldUpdates.status === _task.TaskStatus.Failed || fieldUpdates.status === _task.TaskStatus.ShouldDelete) {
      // Delete the SO instead so it doesn't remain in the index forever
      this.instance = asRan(this.instance.task);
      await this.removeTask();
    } else {
      const {
        shouldValidate = true
      } = (0, _result_type.unwrap)(result);
      let shouldUpdateTask = false;
      let partialTask = {
        id: this.instance.task.id,
        version: this.instance.task.version
      };
      if (this.isExpired) {
        var _this$usageCounter;
        (_this$usageCounter = this.usageCounter) === null || _this$usageCounter === void 0 ? void 0 : _this$usageCounter.incrementCounter({
          counterName: `taskManagerUpdateSkippedDueToTaskExpiration`,
          counterType: 'taskManagerTaskRunner',
          incrementBy: 1
        });
        if (this.shouldUpdateExpiredTask()) {
          shouldUpdateTask = true;
          partialTask = {
            ...partialTask,
            // excluding updated task state from the update
            // because the execution ended in failure due to timeout, we do not update the state
            status: _task.TaskStatus.Idle,
            runAt: fieldUpdates.runAt,
            // reset fields that track the lifecycle of the concluded `task run`
            startedAt: null,
            retryAt: null,
            ownerId: null
          };
        }
      } else {
        shouldUpdateTask = true;
        if (shouldTaskBeDisabled) {
          const label = `${this.taskType}:${this.instance.task.id}`;
          this.logger.warn(`Disabling task ${label} as it indicated it should disable itself`, {
            tags: [this.taskType]
          });
        }
        partialTask = {
          ...partialTask,
          ...fieldUpdates,
          // reset fields that track the lifecycle of the concluded `task run`
          startedAt: null,
          retryAt: null,
          ownerId: null,
          ...(shouldTaskBeDisabled ? {
            enabled: false
          } : {})
        };
      }
      if (shouldUpdateTask) {
        this.instance = asRan(await this.bufferedTaskStore.partialUpdate(partialTask, {
          validate: shouldValidate,
          doc: this.instance.task
        }));
      }
    }
    return fieldUpdates.status === _task.TaskStatus.Failed ? TaskRunResult.Failed : fieldUpdates.status === _task.TaskStatus.ShouldDelete ? TaskRunResult.Deleted : hasTaskRunFailed ? TaskRunResult.SuccessRescheduled : TaskRunResult.RetryScheduled;
  }
  async processResultWhenDone() {
    // not a recurring task: clean up by removing the task instance from store
    try {
      this.instance = asRan(this.instance.task);
      await this.removeTask();
    } catch (err) {
      if (err.statusCode === 404) {
        this.logger.warn(`Task cleanup of ${this} failed in processing. Was remove called twice?`);
      } else {
        throw err;
      }
    }
    return TaskRunResult.Success;
  }
  async processResult(result, taskTiming) {
    const {
      task
    } = this.instance;
    const debugLogger = (0, _wrapped_logger.createWrappedLogger)({
      logger: this.logger,
      tags: [`metrics-debugger`]
    });
    const taskHasExpired = this.isExpired;
    await (0, _result_type.eitherAsync)(result, async ({
      runAt,
      schedule,
      taskRunError
    }) => {
      const taskPersistence = schedule || task.schedule ? _task_events.TaskPersistence.Recurring : _task_events.TaskPersistence.NonRecurring;
      try {
        const processedResult = {
          task,
          persistence: taskPersistence,
          result: await (runAt || schedule || task.schedule ? this.processResultForRecurringTask(result) : this.processResultWhenDone())
        };

        // Alerting task runner returns SuccessfulRunResult with taskRunError
        // when the alerting task fails, so we check for this condition in order
        // to emit the correct task run event for metrics collection
        // taskRunError contains the "source" (TaskErrorSource) data
        if (!!taskRunError) {
          debugLogger.debug(`Emitting task run failed event for task ${this.taskType}`);
          this.onTaskEvent((0, _task_events.asTaskRunEvent)(this.id, (0, _result_type.asErr)({
            ...processedResult,
            isExpired: taskHasExpired,
            error: taskRunError
          }), taskTiming));
        } else {
          this.onTaskEvent((0, _task_events.asTaskRunEvent)(this.id, (0, _result_type.asOk)({
            ...processedResult,
            isExpired: taskHasExpired
          }), taskTiming));
        }
      } catch (err) {
        this.onTaskEvent((0, _task_events.asTaskRunEvent)(this.id, (0, _result_type.asErr)({
          task,
          persistence: taskPersistence,
          result: TaskRunResult.Failed,
          isExpired: taskHasExpired,
          error: err
        }), taskTiming));
        throw err;
      }
    }, async ({
      error
    }) => {
      debugLogger.debug(`Emitting task run failed event for task ${this.taskType}`);
      this.onTaskEvent((0, _task_events.asTaskRunEvent)(this.id, (0, _result_type.asErr)({
        task,
        persistence: task.schedule ? _task_events.TaskPersistence.Recurring : _task_events.TaskPersistence.NonRecurring,
        result: await this.processResultForRecurringTask(result),
        isExpired: taskHasExpired,
        error
      }), taskTiming));
    });
    const {
      eventLoopBlockMs = 0
    } = taskTiming;
    const taskLabel = `${this.taskType} ${this.instance.task.id}`;
    if (eventLoopBlockMs > this.config.event_loop_delay.warn_threshold) {
      this.logger.warn(`event loop blocked for at least ${eventLoopBlockMs} ms while running task ${taskLabel}`, {
        tags: [this.taskType, taskLabel, 'event-loop-blocked']
      });
    }
    return result;
  }
  getMaxAttempts() {
    var _this$definition$maxA, _this$definition;
    return (_this$definition$maxA = (_this$definition = this.definition) === null || _this$definition === void 0 ? void 0 : _this$definition.maxAttempts) !== null && _this$definition$maxA !== void 0 ? _this$definition$maxA : this.defaultMaxAttempts;
  }
  getFakeKibanaRequest(apiKey, spaceId) {
    if (apiKey) {
      const requestHeaders = {};
      requestHeaders.authorization = `ApiKey ${apiKey}`;
      const path = (0, _spacesUtils.addSpaceIdToPath)('/', spaceId || 'default');
      const fakeRawRequest = {
        headers: requestHeaders,
        path: '/'
      };
      const fakeRequest = (0, _coreHttpServerUtils.kibanaRequestFactory)(fakeRawRequest);
      this.basePathService.set(fakeRequest, path);
      return fakeRequest;
    }
  }
}
exports.TaskManagerRunner = TaskManagerRunner;
function sanitizeInstance(instance) {
  return {
    ...instance,
    params: instance.params || {},
    state: instance.state || {}
  };
}
function howManyMsUntilOwnershipClaimExpires(ownershipClaimedUntil) {
  return ownershipClaimedUntil ? ownershipClaimedUntil.getTime() - Date.now() : 0;
}

// Omits "enabled" field from task updates so we don't overwrite any user
// initiated changes to "enabled" while the task was running
function taskWithoutEnabled(task) {
  return (0, _lodash.omit)(task, 'enabled');
}

// A type that extracts the Instance type out of TaskRunningStage
// This helps us to better communicate to the developer what the expected "stage"
// in a specific place in the code might be

function isPending(taskRunning) {
  return taskRunning.stage === TaskRunningStage.PENDING;
}
function asPending(task) {
  return {
    timestamp: new Date(),
    stage: TaskRunningStage.PENDING,
    task
  };
}
function isReadyToRun(taskRunning) {
  return taskRunning.stage === TaskRunningStage.READY_TO_RUN;
}
function asReadyToRun(task) {
  return {
    timestamp: new Date(),
    stage: TaskRunningStage.READY_TO_RUN,
    task
  };
}
function asRan(task) {
  return {
    timestamp: new Date(),
    stage: TaskRunningStage.RAN,
    task
  };
}
function getTaskDelayInSeconds(scheduledAt) {
  const now = new Date();
  return (now.valueOf() - scheduledAt.valueOf()) / 1000;
}