"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TaskPollingLifecycle = void 0;
exports.claimAvailableTasks = claimAvailableTasks;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _pipeable = require("fp-ts/lib/pipeable");
var _Option = require("fp-ts/lib/Option");
var _result_type = require("./lib/result_type");
var _config = require("./config");
var _task_events = require("./task_events");
var _fill_pool = require("./lib/fill_pool");
var _intervals = require("./lib/intervals");
var _polling = require("./polling");
var _task_pool = require("./task_pool");
var _task_running = require("./task_running");
var _identify_es_error = require("./lib/identify_es_error");
var _buffered_task_store = require("./buffered_task_store");
var _task_claiming = require("./queries/task_claiming");
var _create_managed_configuration = require("./lib/create_managed_configuration");
var _task_run_calculators = require("./monitoring/task_run_calculators");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_BUFFER_OPERATIONS = 100;
/**
 * The public interface into the task manager system.
 */
class TaskPollingLifecycle {
  /**
   * Initializes the task manager, preventing any further addition of middleware,
   * enabling the task manipulation methods, and beginning the background polling
   * mechanism.
   */
  constructor({
    basePathService,
    logger,
    middleware,
    config,
    // Elasticsearch and SavedObjects availability status
    elasticsearchAndSOAvailability$,
    taskStore,
    definitions,
    executionContext,
    usageCounter,
    taskPartitioner,
    startingCapacity
  }) {
    (0, _defineProperty2.default)(this, "definitions", void 0);
    (0, _defineProperty2.default)(this, "store", void 0);
    (0, _defineProperty2.default)(this, "taskClaiming", void 0);
    (0, _defineProperty2.default)(this, "bufferedStore", void 0);
    (0, _defineProperty2.default)(this, "basePathService", void 0);
    (0, _defineProperty2.default)(this, "executionContext", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "poller", void 0);
    (0, _defineProperty2.default)(this, "started", false);
    (0, _defineProperty2.default)(this, "pool", void 0);
    (0, _defineProperty2.default)(this, "capacityConfiguration$", void 0);
    (0, _defineProperty2.default)(this, "pollIntervalConfiguration$", void 0);
    // all task related events (task claimed, task marked as running, etc.) are emitted through events$
    (0, _defineProperty2.default)(this, "events$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "middleware", void 0);
    (0, _defineProperty2.default)(this, "usageCounter", void 0);
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "currentPollInterval", void 0);
    (0, _defineProperty2.default)(this, "currentTmUtilization$", new _rxjs.BehaviorSubject(0));
    (0, _defineProperty2.default)(this, "emitEvent", event => {
      this.events$.next(event);
    });
    (0, _defineProperty2.default)(this, "createTaskRunnerForTask", instance => {
      return new _task_running.TaskManagerRunner({
        basePathService: this.basePathService,
        logger: this.logger,
        instance,
        store: this.bufferedStore,
        definitions: this.definitions,
        beforeRun: this.middleware.beforeRun,
        beforeMarkRunning: this.middleware.beforeMarkRunning,
        onTaskEvent: this.emitEvent,
        defaultMaxAttempts: this.taskClaiming.maxAttempts,
        executionContext: this.executionContext,
        usageCounter: this.usageCounter,
        config: this.config,
        allowReadingInvalidState: this.config.allow_reading_invalid_state,
        strategy: this.config.claim_strategy,
        getPollInterval: () => this.currentPollInterval
      });
    });
    (0, _defineProperty2.default)(this, "pollForWork", async () => {
      return (0, _fill_pool.fillPool)(
      // claim available tasks
      async () => {
        const result = await claimAvailableTasks(this.taskClaiming, this.logger);
        if ((0, _result_type.isOk)(result) && result.value.timing) {
          this.emitEvent((0, _task_events.asTaskManagerStatEvent)('claimDuration', (0, _result_type.asOk)(result.value.timing.stop - result.value.timing.start)));
        }
        return result;
      },
      // wrap each task in a Task Runner
      this.createTaskRunnerForTask,
      // place tasks in the Task Pool
      async tasks => {
        const tasksToRun = [];
        const removeTaskPromises = [];
        for (const task of tasks) {
          if (task.isAdHocTaskAndOutOfAttempts) {
            this.logger.debug(`Removing ${task} because the max attempts have been reached.`);
            removeTaskPromises.push(task.removeTask());
          } else {
            tasksToRun.push(task);
          }
        }
        // Wait for all the promises at once to speed up the polling cycle
        const [result] = await Promise.all([this.pool.run(tasksToRun), ...removeTaskPromises]);
        // Emit the load after fetching tasks, giving us a good metric for evaluating how
        // busy Task manager tends to be in this Kibana instance
        this.emitEvent((0, _task_events.asTaskManagerStatEvent)('load', (0, _result_type.asOk)(this.pool.usedCapacityPercentage)));
        return result;
      });
    });
    this.basePathService = basePathService;
    this.logger = logger;
    this.middleware = middleware;
    this.definitions = definitions;
    this.store = taskStore;
    this.executionContext = executionContext;
    this.usageCounter = usageCounter;
    this.config = config;
    const {
      poll_interval: pollInterval,
      claim_strategy: claimStrategy
    } = config;
    this.currentPollInterval = pollInterval;
    const errorCheck$ = (0, _create_managed_configuration.countErrors)(taskStore.errors$, _create_managed_configuration.ADJUST_THROUGHPUT_INTERVAL);
    const window = _config.WORKER_UTILIZATION_RUNNING_AVERAGE_WINDOW_SIZE_MS / this.currentPollInterval;
    const tmUtilizationQueue = (0, _task_run_calculators.createRunningAveragedStat)(window);
    this.capacityConfiguration$ = errorCheck$.pipe((0, _create_managed_configuration.createCapacityScan)(config, logger, startingCapacity), (0, _rxjs.startWith)(startingCapacity), (0, _rxjs.distinctUntilChanged)());
    this.pollIntervalConfiguration$ = errorCheck$.pipe((0, _rxjs.withLatestFrom)(this.currentTmUtilization$), (0, _create_managed_configuration.createPollIntervalScan)(logger, this.currentPollInterval, claimStrategy, tmUtilizationQueue), (0, _rxjs.startWith)(this.currentPollInterval), (0, _rxjs.distinctUntilChanged)());
    this.pollIntervalConfiguration$.subscribe(newPollInterval => {
      this.currentPollInterval = newPollInterval;
    });
    const emitEvent = event => this.events$.next(event);
    this.bufferedStore = new _buffered_task_store.BufferedTaskStore(this.store, {
      bufferMaxOperations: MAX_BUFFER_OPERATIONS,
      logger
    });
    this.pool = new _task_pool.TaskPool({
      logger,
      strategy: config.claim_strategy,
      capacity$: this.capacityConfiguration$,
      definitions: this.definitions
    });
    this.pool.load.subscribe(emitEvent);
    this.taskClaiming = new _task_claiming.TaskClaiming({
      taskStore,
      strategy: config.claim_strategy,
      maxAttempts: config.max_attempts,
      excludedTaskTypes: config.unsafe.exclude_task_types,
      definitions,
      logger: this.logger,
      getAvailableCapacity: taskType => this.pool.availableCapacity(taskType),
      taskPartitioner
    });
    // pipe taskClaiming events into the lifecycle event stream
    this.taskClaiming.events.subscribe(emitEvent);
    let pollIntervalDelay$;
    if (claimStrategy === _config.CLAIM_STRATEGY_UPDATE_BY_QUERY) {
      pollIntervalDelay$ = (0, _polling.delayOnClaimConflicts)(this.capacityConfiguration$, this.pollIntervalConfiguration$, this.events$, config.version_conflict_threshold, config.monitored_stats_running_average_window).pipe((0, _rxjs.tap)(delay => emitEvent((0, _task_events.asTaskManagerStatEvent)('pollingDelay', (0, _result_type.asOk)(delay)))));
    }
    this.poller = (0, _polling.createTaskPoller)({
      logger,
      initialPollInterval: pollInterval,
      pollInterval$: this.pollIntervalConfiguration$,
      pollIntervalDelay$,
      getCapacity: () => {
        const capacity = this.pool.availableCapacity();
        if (!capacity) {
          const usedCapacityPercentage = this.pool.usedCapacityPercentage;

          // if there isn't capacity, emit a load event so that we can expose how often
          // high load causes the poller to skip work (work isn't called when there is no capacity)
          this.emitEvent((0, _task_events.asTaskManagerStatEvent)('load', (0, _result_type.asOk)(usedCapacityPercentage)));

          // Emit event indicating task manager utilization
          this.emitEvent((0, _task_events.asTaskManagerStatEvent)('workerUtilization', (0, _result_type.asOk)(usedCapacityPercentage)));
        }
        return capacity;
      },
      work: this.pollForWork
    });
    this.subscribeToPoller(this.poller.events$);
    elasticsearchAndSOAvailability$.subscribe(areESAndSOAvailable => {
      if (areESAndSOAvailable && !this.started) {
        this.poller.start();
        this.started = true;
      }
    });
  }
  get events() {
    return this.events$;
  }
  stop() {
    this.poller.stop();
  }
  subscribeToPoller(poller$) {
    return poller$.pipe((0, _rxjs.tap)((0, _result_type.mapErr)(error => {
      if (error.type === _polling.PollingErrorType.RequestCapacityReached) {
        (0, _pipeable.pipe)(error.data, (0, _Option.map)(id => this.emitEvent((0, _task_events.asTaskRunRequestEvent)(id, (0, _result_type.asErr)(error)))));
      }
      this.logger.error(error.message, {
        error: {
          stack_trace: error.stack
        }
      });

      // Emit event indicating task manager utilization % at the end of a polling cycle
      // Because there was a polling error, no tasks were claimed so this represents the number of workers busy
      this.emitEvent((0, _task_events.asTaskManagerStatEvent)('workerUtilization', (0, _result_type.asOk)(this.pool.usedCapacityPercentage)));
    }))).pipe((0, _rxjs.tap)((0, _result_type.mapOk)(results => {
      var _results$stats$tasksL, _results$stats;
      // Emit event indicating task manager utilization % at the end of a polling cycle

      // Get the actual utilization as a percentage
      let tmUtilization = this.pool.usedCapacityPercentage;

      // Check whether there are any tasks left unclaimed
      // If we're not at capacity and there are unclaimed tasks, then
      // there must be high cost tasks that need to be claimed
      // Artificially inflate the utilization to represent the unclaimed load
      if (tmUtilization < 100 && ((_results$stats$tasksL = (_results$stats = results.stats) === null || _results$stats === void 0 ? void 0 : _results$stats.tasksLeftUnclaimed) !== null && _results$stats$tasksL !== void 0 ? _results$stats$tasksL : 0) > 0) {
        tmUtilization = 100;
      }
      this.currentTmUtilization$.next(tmUtilization);
      this.emitEvent((0, _task_events.asTaskManagerStatEvent)('workerUtilization', (0, _result_type.asOk)(tmUtilization)));
    }))).subscribe(result => {
      this.emitEvent((0, _result_type.map)(result, ({
        timing,
        ...event
      }) => {
        var _event$stats$tasksErr, _event$stats;
        const anyTaskErrors = (_event$stats$tasksErr = (_event$stats = event.stats) === null || _event$stats === void 0 ? void 0 : _event$stats.tasksErrors) !== null && _event$stats$tasksErr !== void 0 ? _event$stats$tasksErr : 0;
        if (anyTaskErrors > 0) {
          return (0, _task_events.asTaskPollingCycleEvent)((0, _result_type.asErr)(new _polling.PollingError('Partially failed to poll for work: some tasks could not be claimed.', _polling.PollingErrorType.WorkError, _Option.none)));
        }
        return (0, _task_events.asTaskPollingCycleEvent)((0, _result_type.asOk)(event), timing);
      }, event => (0, _task_events.asTaskPollingCycleEvent)((0, _result_type.asErr)(event))));
    });
  }
}
exports.TaskPollingLifecycle = TaskPollingLifecycle;
async function claimAvailableTasks(taskClaiming, logger) {
  try {
    return taskClaiming.claimAvailableTasksIfCapacityIsAvailable({
      claimOwnershipUntil: (0, _intervals.intervalFromNow)('30s')
    });
  } catch (err) {
    // if we can identify the reason for the error, emit a FillPoolResult error
    if ((0, _identify_es_error.isEsCannotExecuteScriptError)(err)) {
      logger.warn(`Task Manager cannot operate when inline scripts are disabled in Elasticsearch`);
      return (0, _result_type.asErr)(_fill_pool.FillPoolResult.Failed);
    } else {
      const esError = (0, _identify_es_error.identifyEsError)(err);
      // as we could't identify the reason - propagate the error
      throw esError.length > 0 ? esError : err;
    }
  }
}