"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ProjectMonitorFormatter = exports.FAILED_TO_UPDATE_MONITOR = exports.CANNOT_UPDATE_MONITOR_TO_DIFFERENT_TYPE = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _i18n = require("@kbn/i18n");
var _saved_objects = require("../../../common/types/saved_objects");
var _common = require("../../routes/common");
var _common_fields = require("./normalizers/common_fields");
var _get_all_locations = require("../get_all_locations");
var _add_monitor_bulk = require("../../routes/monitor_cruds/bulk_cruds/add_monitor_bulk");
var _edit_monitor_bulk = require("../../routes/monitor_cruds/bulk_cruds/edit_monitor_bulk");
var _runtime_types = require("../../../common/runtime_types");
var _secrets = require("../utils/secrets");
var _monitor_validation = require("../../routes/monitor_cruds/monitor_validation");
var _normalizers = require("./normalizers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CANNOT_UPDATE_MONITOR_TO_DIFFERENT_TYPE = exports.CANNOT_UPDATE_MONITOR_TO_DIFFERENT_TYPE = _i18n.i18n.translate('xpack.synthetics.service.projectMonitors.cannotUpdateMonitorToDifferentType', {
  defaultMessage: 'Cannot update monitor to different type.'
});
const FAILED_TO_UPDATE_MONITOR = exports.FAILED_TO_UPDATE_MONITOR = _i18n.i18n.translate('xpack.synthetics.service.projectMonitors.failedToUpdateMonitor', {
  defaultMessage: 'Failed to create or update monitor'
});
class ProjectMonitorFormatter {
  constructor({
    projectId,
    spaceId,
    monitors: _monitors,
    routeContext
  }) {
    (0, _defineProperty2.default)(this, "projectId", void 0);
    (0, _defineProperty2.default)(this, "spaceId", void 0);
    (0, _defineProperty2.default)(this, "publicLocations", void 0);
    (0, _defineProperty2.default)(this, "privateLocations", void 0);
    (0, _defineProperty2.default)(this, "savedObjectsClient", void 0);
    (0, _defineProperty2.default)(this, "monitors", []);
    (0, _defineProperty2.default)(this, "createdMonitors", []);
    (0, _defineProperty2.default)(this, "updatedMonitors", []);
    (0, _defineProperty2.default)(this, "failedMonitors", []);
    (0, _defineProperty2.default)(this, "server", void 0);
    (0, _defineProperty2.default)(this, "projectFilter", void 0);
    (0, _defineProperty2.default)(this, "syntheticsMonitorClient", void 0);
    (0, _defineProperty2.default)(this, "routeContext", void 0);
    (0, _defineProperty2.default)(this, "init", async () => {
      const locationsPromise = (0, _get_all_locations.getAllLocations)({
        server: this.server,
        syntheticsMonitorClient: this.syntheticsMonitorClient,
        savedObjectsClient: this.savedObjectsClient,
        excludeAgentPolicies: true
      });
      const existingMonitorsPromise = this.getProjectMonitorsForProject();
      const [locations, existingMonitors] = await Promise.all([locationsPromise, existingMonitorsPromise]);
      const {
        publicLocations,
        privateLocations
      } = locations;
      this.publicLocations = publicLocations;
      this.privateLocations = privateLocations;
      return existingMonitors;
    });
    (0, _defineProperty2.default)(this, "configureAllProjectMonitors", async () => {
      const existingMonitors = await this.init();
      const normalizedNewMonitors = [];
      const normalizedUpdateMonitors = [];
      for (const monitor of this.monitors) {
        const previousMonitor = existingMonitors.find(monitorObj => monitorObj[_runtime_types.ConfigKey.JOURNEY_ID] === monitor.id);
        const normM = this.validateProjectMonitor({
          monitor,
          publicLocations: this.publicLocations,
          privateLocations: this.privateLocations
        });
        if (normM) {
          if (previousMonitor && previousMonitor[_runtime_types.ConfigKey.MONITOR_TYPE] !== normM[_runtime_types.ConfigKey.MONITOR_TYPE]) {
            this.failedMonitors.push({
              reason: CANNOT_UPDATE_MONITOR_TO_DIFFERENT_TYPE,
              details: _i18n.i18n.translate('xpack.synthetics.service.projectMonitors.cannotUpdateMonitorToDifferentTypeDetails', {
                defaultMessage: 'Monitor {monitorId} of type {previousType} cannot be updated to type {currentType}. Please delete the monitor first and try again.',
                values: {
                  currentType: monitor.type,
                  previousType: previousMonitor[_runtime_types.ConfigKey.MONITOR_TYPE],
                  monitorId: monitor.id
                }
              }),
              payload: monitor
            });
          } else if (previousMonitor) {
            this.updatedMonitors.push(monitor.id);
            normalizedUpdateMonitors.push({
              monitor: normM,
              previousMonitor
            });
          } else {
            normalizedNewMonitors.push(normM);
          }
        }
      }
      await Promise.allSettled([this.createMonitorsBulk(normalizedNewMonitors), this.updateMonitorsBulk(normalizedUpdateMonitors)]);
    });
    (0, _defineProperty2.default)(this, "validateProjectMonitor", ({
      monitor,
      publicLocations,
      privateLocations
    }) => {
      try {
        const {
          normalizedFields: normalizedMonitor,
          errors
        } = (0, _normalizers.normalizeProjectMonitor)({
          monitor,
          locations: this.publicLocations,
          privateLocations: this.privateLocations,
          projectId: this.projectId,
          namespace: this.spaceId,
          version: this.server.stackVersion
        });
        if (errors.length) {
          this.failedMonitors.push(...errors);
          return null;
        }

        /* Validates that the payload sent from the synthetics agent is valid */
        const {
          valid: isMonitorPayloadValid
        } = this.validateMonitor({
          validationResult: (0, _monitor_validation.validateProjectMonitor)({
            ...monitor,
            type: normalizedMonitor[_runtime_types.ConfigKey.MONITOR_TYPE]
          }, publicLocations, privateLocations),
          monitorId: monitor.id
        });
        if (!isMonitorPayloadValid) {
          return null;
        }

        /* Validates that the normalized monitor is a valid monitor saved object type */
        const {
          valid: isNormalizedMonitorValid,
          decodedMonitor
        } = this.validateMonitor({
          validationResult: (0, _monitor_validation.validateMonitor)(normalizedMonitor, this.spaceId),
          monitorId: monitor.id
        });
        if (!isNormalizedMonitorValid || !decodedMonitor) {
          return null;
        }
        return decodedMonitor;
      } catch (e) {
        this.server.logger.error(e);
        const reason = e instanceof _common_fields.InvalidLocationError ? _monitor_validation.INVALID_CONFIGURATION_ERROR : FAILED_TO_UPDATE_MONITOR;
        this.failedMonitors.push({
          reason,
          id: monitor.id,
          details: e.message,
          payload: monitor
        });
      }
    });
    (0, _defineProperty2.default)(this, "getProjectMonitorsForProject", async () => {
      const journeyIds = this.monitors.map(monitor => monitor.id);
      const journeyFilter = (0, _common.getSavedObjectKqlFilter)({
        field: _runtime_types.ConfigKey.JOURNEY_ID,
        values: journeyIds
      });
      const result = await this.routeContext.monitorConfigRepository.find({
        filter: `${this.projectFilter} AND ${journeyFilter}`,
        fields: [_runtime_types.ConfigKey.JOURNEY_ID, _runtime_types.ConfigKey.CONFIG_ID, _runtime_types.ConfigKey.REVISION, _runtime_types.ConfigKey.MONITOR_TYPE]
      });
      return result.saved_objects.map(monitor => {
        return {
          ...monitor.attributes,
          updated_at: monitor.updated_at
        };
      });
    });
    (0, _defineProperty2.default)(this, "createMonitorsBulk", async monitors => {
      try {
        if (monitors.length > 0) {
          const {
            newMonitors,
            failedMonitors
          } = await (0, _add_monitor_bulk.syncNewMonitorBulk)({
            normalizedMonitors: monitors,
            routeContext: this.routeContext,
            privateLocations: this.privateLocations,
            spaceId: this.spaceId
          });
          if (newMonitors.length > 0) {
            newMonitors.forEach(monitor => {
              const journeyId = monitor.attributes[_runtime_types.ConfigKey.JOURNEY_ID];
              if (journeyId && !monitor.error) {
                this.createdMonitors.push(journeyId);
              } else if (monitor.error) {
                this.failedMonitors.push({
                  reason: _i18n.i18n.translate('xpack.synthetics.service.projectMonitors.failedToCreateMonitors', {
                    defaultMessage: 'Failed to create monitor: {journeyId}',
                    values: {
                      journeyId
                    }
                  }),
                  details: monitor.error.message,
                  payload: monitor
                });
              }
            });
          }
          failedMonitors.forEach(({
            monitor,
            error
          }) => {
            var _error$message;
            const journeyId = monitor.attributes[_runtime_types.ConfigKey.JOURNEY_ID];
            this.failedMonitors.push({
              reason: (_error$message = error === null || error === void 0 ? void 0 : error.message) !== null && _error$message !== void 0 ? _error$message : FAILED_TO_UPDATE_MONITOR,
              details: _i18n.i18n.translate('xpack.synthetics.service.projectMonitors.failedToCreateMonitors', {
                defaultMessage: 'Failed to create monitor: {journeyId}',
                values: {
                  journeyId
                }
              }),
              payload: monitors
            });
          });
        }
      } catch (e) {
        this.server.logger.error(e);
        this.failedMonitors.push({
          reason: _i18n.i18n.translate('xpack.synthetics.service.projectMonitors.failedToCreateXMonitors', {
            defaultMessage: 'Failed to create {length} monitors',
            values: {
              length: monitors.length
            }
          }),
          details: e.message,
          payload: monitors
        });
      }
    });
    (0, _defineProperty2.default)(this, "getDecryptedMonitors", async monitors => {
      const configIds = monitors.map(monitor => monitor[_runtime_types.ConfigKey.CONFIG_ID]);
      const monitorFilter = (0, _common.getSavedObjectKqlFilter)({
        field: _runtime_types.ConfigKey.CONFIG_ID,
        values: configIds
      });
      return await this.routeContext.monitorConfigRepository.findDecryptedMonitors({
        filter: monitorFilter,
        spaceId: this.spaceId
      });
    });
    (0, _defineProperty2.default)(this, "updateMonitorsBulk", async monitors => {
      try {
        if (monitors.length === 0) {
          return {
            editedMonitors: [],
            errors: [],
            updatedCount: 0
          };
        }
        const decryptedPreviousMonitors = await this.getDecryptedMonitors(monitors.map(m => m.previousMonitor));
        const monitorsToUpdate = [];
        decryptedPreviousMonitors.forEach(decryptedPreviousMonitor => {
          const monitor = monitors.find(m => m.previousMonitor[_runtime_types.ConfigKey.CONFIG_ID] === decryptedPreviousMonitor.id);
          if (monitor) {
            const normalizedMonitor = monitor === null || monitor === void 0 ? void 0 : monitor.monitor;
            const {
              attributes: {
                [_runtime_types.ConfigKey.REVISION]: _,
                ...normalizedPrevMonitorAttr
              }
            } = (0, _secrets.normalizeSecrets)(decryptedPreviousMonitor);
            const monitorWithRevision = (0, _secrets.formatSecrets)({
              ...normalizedPrevMonitorAttr,
              ...normalizedMonitor,
              revision: (decryptedPreviousMonitor.attributes[_runtime_types.ConfigKey.REVISION] || 0) + 1
            });
            monitorsToUpdate.push({
              normalizedMonitor,
              monitorWithRevision,
              decryptedPreviousMonitor
            });
          }
        });
        const {
          editedMonitors,
          failedConfigs
        } = await (0, _edit_monitor_bulk.syncEditedMonitorBulk)({
          monitorsToUpdate,
          routeContext: this.routeContext,
          privateLocations: this.privateLocations,
          spaceId: this.spaceId
        });
        if (failedConfigs && Object.keys(failedConfigs).length > 0) {
          const failedConfigsIds = Object.keys(failedConfigs);
          failedConfigsIds.forEach(id => {
            var _error$message2;
            const {
              config,
              error
            } = failedConfigs[id];
            const journeyId = config[_runtime_types.ConfigKey.JOURNEY_ID];
            this.failedMonitors.push({
              reason: (_error$message2 = error === null || error === void 0 ? void 0 : error.message) !== null && _error$message2 !== void 0 ? _error$message2 : FAILED_TO_UPDATE_MONITOR,
              details: _i18n.i18n.translate('xpack.synthetics.service.projectMonitors.failedToUpdateJourney', {
                defaultMessage: 'Failed to update journey: {journeyId}',
                values: {
                  journeyId
                }
              }),
              payload: config
            });
          });

          // remove failed monitors from the list of updated monitors
          this.updatedMonitors.splice(this.updatedMonitors.findIndex(monitorId => failedConfigsIds.includes(monitorId)), failedConfigsIds.length);
        }
        return {
          errors: [],
          editedMonitors: editedMonitors !== null && editedMonitors !== void 0 ? editedMonitors : [],
          updatedCount: monitorsToUpdate.length
        };
      } catch (e) {
        this.server.logger.error(e);
        this.failedMonitors.push({
          reason: _i18n.i18n.translate('xpack.synthetics.service.projectMonitors.failedToUpdateXMonitors', {
            defaultMessage: 'Failed to update {length} monitors',
            values: {
              length: monitors.length
            }
          }),
          details: e.message,
          payload: monitors
        });
      }
    });
    (0, _defineProperty2.default)(this, "validateMonitor", ({
      validationResult,
      monitorId
    }) => {
      const {
        reason: message,
        details,
        payload: validationPayload,
        valid
      } = validationResult;
      if (!valid) {
        this.failedMonitors.push({
          id: monitorId,
          reason: message,
          details,
          payload: validationPayload
        });
      }
      return validationResult;
    });
    this.routeContext = routeContext;
    this.projectId = projectId;
    this.spaceId = spaceId;
    this.savedObjectsClient = routeContext.savedObjectsClient;
    this.syntheticsMonitorClient = routeContext.syntheticsMonitorClient;
    this.monitors = _monitors;
    this.server = routeContext.server;
    this.projectFilter = `${_saved_objects.syntheticsMonitorSavedObjectType}.attributes.${_runtime_types.ConfigKey.PROJECT_ID}: "${this.projectId}"`;
    this.publicLocations = [];
    this.privateLocations = [];
  }
}
exports.ProjectMonitorFormatter = ProjectMonitorFormatter;