"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.normalizeAPIConfig = exports.MonitorValidationError = exports.LOCATION_REQUIRED_ERROR = exports.INVALID_CONFIGURATION_ERROR = void 0;
exports.validateLocation = validateLocation;
exports.validateMonitor = validateMonitor;
exports.validateProjectMonitor = validateProjectMonitor;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var t = _interopRequireWildcard(require("io-ts"));
var _i18n = require("@kbn/i18n");
var _Either = require("fp-ts/lib/Either");
var _securitysolutionIoTsUtils = require("@kbn/securitysolution-io-ts-utils");
var _lodash = require("lodash");
var _configSchema = require("@kbn/config-schema");
var _alert_config_schema = require("../../../common/runtime_types/monitor_management/alert_config_schema");
var _common_fields = require("../../synthetics_service/project_monitor/normalizers/common_fields");
var _runtime_types = require("../../../common/runtime_types");
var _monitor_defaults = require("../../../common/constants/monitor_defaults");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const monitorTypeToCodecMap = {
  [_runtime_types.MonitorTypeEnum.ICMP]: _runtime_types.ICMPFieldsCodec,
  [_runtime_types.MonitorTypeEnum.TCP]: _runtime_types.TCPFieldsCodec,
  [_runtime_types.MonitorTypeEnum.HTTP]: _runtime_types.HTTPFieldsCodec,
  [_runtime_types.MonitorTypeEnum.BROWSER]: _runtime_types.BrowserFieldsCodec
};
class MonitorValidationError extends Error {
  constructor(result) {
    super(result.reason);
    (0, _defineProperty2.default)(this, "result", void 0);
    this.result = result;
  }
}

/**
 * Validates monitor fields with respect to the relevant Codec identified by object's 'type' property.
 * @param monitorFields {MonitorFields} The mixed type representing the possible monitor payload from UI.
 * @param spaceId
 */
exports.MonitorValidationError = MonitorValidationError;
function validateMonitor(monitorFields, spaceId) {
  const {
    [_runtime_types.ConfigKey.MONITOR_TYPE]: monitorType,
    [_runtime_types.ConfigKey.KIBANA_SPACES]: kSpaces
  } = monitorFields;
  if (monitorType !== _runtime_types.MonitorTypeEnum.BROWSER && !monitorFields.name) {
    monitorFields.name = monitorFields.urls || monitorFields.hosts;
  }
  if (monitorFields.locations.length === 0) {
    return {
      valid: false,
      reason: LOCATION_REQUIRED_ERROR,
      details: '',
      payload: monitorFields
    };
  }
  const decodedType = _runtime_types.MonitorTypeCodec.decode(monitorType);
  if ((0, _Either.isLeft)(decodedType)) {
    return {
      valid: false,
      reason: INVALID_TYPE_ERROR,
      details: (0, _securitysolutionIoTsUtils.formatErrors)(decodedType.left).join(' | '),
      payload: monitorFields
    };
  }

  // Cast it to ICMPCodec to satisfy typing. During runtime, correct codec will be used to decode.
  const SyntheticsMonitorCodec = monitorTypeToCodecMap[monitorType];
  if (!SyntheticsMonitorCodec) {
    return {
      valid: false,
      reason: INVALID_PAYLOAD_ERROR,
      details: '',
      payload: monitorFields
    };
  }
  const alert = monitorFields.alert;
  if (alert) {
    try {
      _alert_config_schema.AlertConfigSchema.validate(alert);
    } catch (e) {
      return {
        valid: false,
        reason: 'Invalid alert configuration',
        details: e.message,
        payload: monitorFields
      };
    }
  }
  if (!_monitor_defaults.ALLOWED_SCHEDULES_IN_MINUTES.includes(monitorFields[_runtime_types.ConfigKey.SCHEDULE].number)) {
    return {
      valid: false,
      reason: INVALID_SCHEDULE_ERROR,
      details: INVALID_SCHEDULE_DETAILS(monitorFields[_runtime_types.ConfigKey.SCHEDULE].number),
      payload: monitorFields
    };
  }
  const ExactSyntheticsMonitorCodec = t.exact(SyntheticsMonitorCodec);
  const decodedMonitor = ExactSyntheticsMonitorCodec.decode(monitorFields);
  if ((0, _Either.isLeft)(decodedMonitor)) {
    return {
      valid: false,
      reason: INVALID_SCHEMA_ERROR(monitorType),
      details: (0, _securitysolutionIoTsUtils.formatErrors)(decodedMonitor.left).join(' | '),
      payload: monitorFields
    };
  }
  if (monitorType === _runtime_types.MonitorTypeEnum.BROWSER) {
    const inlineScript = monitorFields[_runtime_types.ConfigKey.SOURCE_INLINE];
    const projectContent = monitorFields[_runtime_types.ConfigKey.SOURCE_PROJECT_CONTENT];
    if (!inlineScript && !projectContent) {
      return {
        valid: false,
        reason: 'Monitor is not a valid monitor of type browser',
        details: _i18n.i18n.translate('xpack.synthetics.createMonitor.validation.noScript', {
          defaultMessage: 'source.inline.script: Script is required for browser monitor.'
        }),
        payload: monitorFields
      };
    }
  }
  if (spaceId && !(0, _lodash.isEmpty)(kSpaces)) {
    // we throw error if kSpaces is not empty and spaceId is not present
    if (kSpaces && !kSpaces.includes(spaceId) && !kSpaces.includes('*')) {
      return {
        valid: false,
        reason: _i18n.i18n.translate('xpack.synthetics.createMonitor.validation.invalidSpace', {
          defaultMessage: 'Invalid space ID provided in monitor configuration. It should always include the current space ID.'
        }),
        details: '',
        payload: monitorFields
      };
    }
  }
  return {
    valid: true,
    reason: '',
    details: '',
    payload: monitorFields,
    decodedMonitor: decodedMonitor.right
  };
}
const normalizeAPIConfig = monitor => {
  var _kSpaces;
  const monitorType = monitor.type;
  const decodedType = _runtime_types.MonitorTypeCodec.decode(monitorType);
  if ((0, _Either.isLeft)(decodedType)) {
    return {
      errorMessage: (0, _securitysolutionIoTsUtils.formatErrors)(decodedType.left).join(' | ')
    };
  }
  const defaultFields = _monitor_defaults.DEFAULT_FIELDS[monitor.type];
  let supportedKeys = Object.keys(defaultFields);
  const flattenedConfig = (0, _common_fields.flattenAndFormatObject)(monitor, '', supportedKeys);
  const {
    locations: _locations,
    private_locations: _privateLocations,
    id: _id,
    retest_on_failure: _retestOnFailure,
    url: rawUrl,
    ssl: _rawSSL,
    host: rawHost,
    inline_script: inlineScript,
    custom_heartbeat_id: _customHeartbeatId,
    params: rawParams,
    playwright_options: rawPlaywrightOptions,
    ...rawConfig
  } = flattenedConfig;
  if (Object.keys(flattenedConfig).some(key => key.startsWith('ssl.')) && monitor.origin !== 'project') {
    var _ref;
    rawConfig[_runtime_types.ConfigKey.METADATA] = {
      is_tls_enabled: true,
      ...((_ref = flattenedConfig[_runtime_types.ConfigKey.METADATA]) !== null && _ref !== void 0 ? _ref : {})
    };
  }
  if (rawUrl) {
    // since api accept url key as well
    rawConfig[_runtime_types.ConfigKey.URLS] = rawUrl;
  }
  if (inlineScript) {
    rawConfig[_runtime_types.ConfigKey.SOURCE_INLINE] = inlineScript;
  }
  if (rawHost) {
    // since api accept url key as well
    rawConfig[_runtime_types.ConfigKey.HOSTS] = rawHost;
  }
  if (monitor.type === 'browser' && monitor[_runtime_types.ConfigKey.FORM_MONITOR_TYPE] !== _runtime_types.FormMonitorType.SINGLE && monitor[_runtime_types.ConfigKey.FORM_MONITOR_TYPE] !== _runtime_types.FormMonitorType.MULTISTEP) {
    // urls isn't supported for browser but is needed for SO AAD
    supportedKeys = supportedKeys.filter(key => key !== _runtime_types.ConfigKey.URLS);
  }
  // needed for SO AAD
  supportedKeys.push(_runtime_types.ConfigKey.PROJECT_ID, _runtime_types.ConfigKey.ORIGINAL_SPACE);
  let unsupportedKeys = Object.keys(rawConfig).filter(key => !supportedKeys.includes(key));
  const result = (0, _lodash.omit)(rawConfig, unsupportedKeys);
  let kSpaces = rawConfig[_runtime_types.ConfigKey.KIBANA_SPACES];
  if ((_kSpaces = kSpaces) !== null && _kSpaces !== void 0 && _kSpaces.includes('*')) {
    kSpaces = ['*'];
  }
  const formattedConfig = {
    ...result,
    locations: _locations,
    private_locations: _privateLocations,
    retest_on_failure: _retestOnFailure,
    custom_heartbeat_id: _customHeartbeatId,
    ...(kSpaces ? {
      [_runtime_types.ConfigKey.KIBANA_SPACES]: kSpaces
    } : {})
  };
  const requestBodyCheck = formattedConfig[_runtime_types.ConfigKey.REQUEST_BODY_CHECK];
  if (typeof requestBodyCheck === 'string') {
    formattedConfig[_runtime_types.ConfigKey.REQUEST_BODY_CHECK] = {
      type: _runtime_types.CodeEditorMode.PLAINTEXT,
      value: requestBodyCheck
    };
  }
  if (rawParams) {
    const {
      value,
      error
    } = validateParams(rawParams);
    if (error) {
      formattedConfig[_runtime_types.ConfigKey.PARAMS] = rawParams;
      return {
        formattedConfig,
        errorMessage: _i18n.i18n.translate('xpack.synthetics.restApi.monitor.invalidParams', {
          defaultMessage: 'Invalid params: {error}',
          values: {
            error: error.message
          }
        })
      };
    }
    formattedConfig[_runtime_types.ConfigKey.PARAMS] = value;
  }
  if (rawPlaywrightOptions) {
    const {
      value,
      error
    } = validateJSON(rawPlaywrightOptions);
    if (error) {
      formattedConfig[_runtime_types.ConfigKey.PLAYWRIGHT_OPTIONS] = rawPlaywrightOptions;
      return {
        formattedConfig,
        errorMessage: _i18n.i18n.translate('xpack.synthetics.restApi.monitor.invalidPlaywrightOptions', {
          defaultMessage: 'Invalid playwright_options: {error}',
          values: {
            error: error.message
          }
        })
      };
    }
    formattedConfig[_runtime_types.ConfigKey.PLAYWRIGHT_OPTIONS] = value;
  }
  if (unsupportedKeys.length > 0) {
    unsupportedKeys = unsupportedKeys.map(key => {
      if (key === _runtime_types.ConfigKey.SOURCE_INLINE && inlineScript) {
        return 'inline_script';
      }
      if (key === _runtime_types.ConfigKey.URLS && rawUrl) {
        return 'url';
      }
      if (key === _runtime_types.ConfigKey.HOSTS && rawHost) {
        return 'host';
      }
      return key;
    });
    return {
      formattedConfig,
      errorMessage: _i18n.i18n.translate('xpack.synthetics.restApi.monitor.invalidMonitorKey', {
        defaultMessage: 'Invalid monitor key(s) for {monitorType} type:  {unsupportedKeys}',
        values: {
          monitorType: monitor.type,
          unsupportedKeys: unsupportedKeys.join(' | ')
        }
      })
    };
  }
  return {
    formattedConfig
  };
};
exports.normalizeAPIConfig = normalizeAPIConfig;
const RecordSchema = _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.string());
const validateParams = jsonString => {
  if (typeof jsonString === 'string') {
    try {
      JSON.parse(jsonString);
      return {
        value: jsonString
      };
    } catch (e) {
      return {
        error: e
      };
    }
  }
  try {
    RecordSchema.validate(jsonString);
    return {
      value: JSON.stringify(jsonString)
    };
  } catch (e) {
    return {
      error: e
    };
  }
};
const validateJSON = jsonString => {
  if (typeof jsonString === 'string') {
    try {
      JSON.parse(jsonString);
      return {
        value: jsonString
      };
    } catch (e) {
      return {
        error: e
      };
    }
  }
  try {
    return {
      value: JSON.stringify(jsonString)
    };
  } catch (e) {
    return {
      error: e
    };
  }
};
function validateProjectMonitor(monitorFields, publicLocations, privateLocations) {
  const locationsError = validateLocation(monitorFields, publicLocations, privateLocations);
  // Cast it to ICMPCodec to satisfy typing. During runtime, correct codec will be used to decode.
  const decodedMonitor = _runtime_types.ProjectMonitorCodec.decode(monitorFields);
  if ((0, _Either.isLeft)(decodedMonitor)) {
    return {
      valid: false,
      reason: INVALID_CONFIGURATION_ERROR,
      details: [...(0, _securitysolutionIoTsUtils.formatErrors)(decodedMonitor.left), locationsError].filter(error => error !== '' && error !== undefined).join(' | '),
      payload: monitorFields
    };
  }
  if (locationsError) {
    return {
      valid: false,
      reason: INVALID_CONFIGURATION_ERROR,
      details: locationsError,
      payload: monitorFields
    };
  }
  return {
    valid: true,
    reason: '',
    details: '',
    payload: monitorFields
  };
}
function validateLocation(monitorFields, publicLocations, privateLocations) {
  var _monitorFields$privat2;
  const hasPublicLocationsConfigured = (monitorFields.locations || []).length > 0;
  const hasPrivateLocationsConfigured = (monitorFields.privateLocations || []).length > 0;
  if (hasPublicLocationsConfigured) {
    var _monitorFields$locati;
    let invalidLocation = '';
    const hasValidPublicLocation = (_monitorFields$locati = monitorFields.locations) === null || _monitorFields$locati === void 0 ? void 0 : _monitorFields$locati.some(location => {
      if (publicLocations.length === 0) {
        invalidLocation = location;
        return false;
      }
      return publicLocations.some(supportedLocation => {
        const locationIsValid = supportedLocation.id === location;
        if (!locationIsValid) {
          invalidLocation = location;
        }
        return locationIsValid;
      });
    });
    if (!hasValidPublicLocation) {
      return INVALID_PUBLIC_LOCATION_ERROR(invalidLocation);
    }
  }
  if (hasPrivateLocationsConfigured) {
    var _monitorFields$privat;
    let invalidLocation = '';
    const hasValidPrivateLocation = (_monitorFields$privat = monitorFields.privateLocations) === null || _monitorFields$privat === void 0 ? void 0 : _monitorFields$privat.some(location => {
      if (privateLocations.length === 0) {
        invalidLocation = location;
        return false;
      }
      return privateLocations.some(supportedLocation => {
        const locationIsValid = supportedLocation.label === location;
        if (!locationIsValid) {
          invalidLocation = location;
        }
        return locationIsValid;
      });
    });
    if (!hasValidPrivateLocation) {
      return INVALID_PRIVATE_LOCATION_ERROR(invalidLocation);
    }
  }
  const hasEmptyLocations = monitorFields.locations && monitorFields.locations.length === 0 && ((_monitorFields$privat2 = monitorFields.privateLocations) !== null && _monitorFields$privat2 !== void 0 ? _monitorFields$privat2 : []).length === 0;
  if (hasEmptyLocations) {
    return EMPTY_LOCATION_ERROR;
  }
}
const INVALID_CONFIGURATION_ERROR = exports.INVALID_CONFIGURATION_ERROR = _i18n.i18n.translate('xpack.synthetics.server.monitors.invalidConfigurationError', {
  defaultMessage: "Couldn't save or update monitor because of an invalid configuration."
});
const INVALID_PAYLOAD_ERROR = _i18n.i18n.translate('xpack.synthetics.server.monitors.invalidPayloadError', {
  defaultMessage: 'Payload is not a valid monitor object'
});
const INVALID_TYPE_ERROR = _i18n.i18n.translate('xpack.synthetics.server.monitors.invalidTypeError', {
  defaultMessage: 'Monitor type is invalid'
});
const INVALID_SCHEDULE_ERROR = _i18n.i18n.translate('xpack.synthetics.server.monitors.invalidScheduleError', {
  defaultMessage: 'Monitor schedule is invalid'
});
const INVALID_SCHEDULE_DETAILS = schedule => _i18n.i18n.translate('xpack.synthetics.server.monitors.invalidScheduleDetails', {
  defaultMessage: 'Invalid schedule {schedule} minutes supplied to monitor configuration. Supported schedule values in minutes are {allowedSchedulesInMinutes}',
  values: {
    schedule,
    allowedSchedulesInMinutes: _monitor_defaults.ALLOWED_SCHEDULES_IN_MINUTES.join(', ')
  }
});
const INVALID_SCHEMA_ERROR = type => _i18n.i18n.translate('xpack.synthetics.server.monitors.invalidSchemaError', {
  defaultMessage: 'Monitor is not a valid monitor of type {type}',
  values: {
    type
  }
});
const EMPTY_LOCATION_ERROR = _i18n.i18n.translate('xpack.synthetics.server.projectMonitors.locationEmptyError', {
  defaultMessage: 'You must add at least one location or private location to this monitor.'
});
const INVALID_PRIVATE_LOCATION_ERROR = location => _i18n.i18n.translate('xpack.synthetics.server.projectMonitors.invalidPrivateLocationError', {
  defaultMessage: 'Invalid private location: "{location}". Remove it or replace it with a valid private location.',
  values: {
    location
  }
});
const INVALID_PUBLIC_LOCATION_ERROR = location => _i18n.i18n.translate('xpack.synthetics.server.projectMonitors.invalidPublicLocationError', {
  defaultMessage: 'Invalid location: "{location}". Remove it or replace it with a valid location.',
  values: {
    location
  }
});
const LOCATION_REQUIRED_ERROR = exports.LOCATION_REQUIRED_ERROR = _i18n.i18n.translate('xpack.synthetics.createMonitor.validation.noLocations', {
  defaultMessage: 'At least one location is required, either elastic managed or private e.g locations: ["us-east"] or private_locations:["test private location"]'
});