"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.contentRoutes = void 0;
var _stream = require("stream");
var _esErrors = require("@kbn/es-errors");
var _zod = require("@kbn/zod");
var _utils = require("@kbn/utils");
var _install = require("@kbn/fleet-plugin/server/services/epm/kibana/assets/install");
var _contentPacksSchema = require("@kbn/content-packs-schema");
var _constants = require("../../../common/constants");
var _create_server_route = require("../create_server_route");
var _status_error = require("../../lib/streams/errors/status_error");
var _fields = require("../../lib/streams/assets/fields");
var _content = require("../../lib/content");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_CONTENT_PACK_SIZE_BYTES = 1024 * 1024 * 5; // 5MB

const exportContentRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /api/streams/{name}/content/export 2023-10-31',
  options: {
    access: 'public',
    summary: 'Export stream content',
    description: 'Exports the content associated to a stream.'
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    body: _zod.z.object({
      name: _zod.z.string(),
      description: _zod.z.string(),
      version: _zod.z.string(),
      replaced_patterns: _zod.z.array(_zod.z.string()),
      include: _contentPacksSchema.contentPackIncludedObjectsSchema
    })
  }),
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  async handler({
    params,
    request,
    response,
    getScopedClients,
    context
  }) {
    const {
      assetClient,
      soClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    await streamsClient.ensureStream(params.path.name);
    if (!(0, _contentPacksSchema.isIncludeAll)(params.body.include) && params.body.include.objects.dashboards.length === 0) {
      throw new _status_error.StatusError(`Content pack must include at least one object`, 400);
    }
    function isDashboard(asset) {
      return asset[_fields.ASSET_TYPE] === 'dashboard';
    }
    const dashboards = (await assetClient.getAssets(params.path.name)).filter(isDashboard).filter(dashboard => (0, _contentPacksSchema.isIncludeAll)(params.body.include) || params.body.include.objects.dashboards.includes(dashboard['asset.id']));
    if (dashboards.length === 0) {
      throw new _status_error.StatusError('No included objects were found', 400);
    }
    const exporter = (await context.core).savedObjects.getExporter(soClient);
    const exportStream = await exporter.exportByObjects({
      request,
      objects: dashboards.map(dashboard => ({
        id: dashboard[_fields.ASSET_ID],
        type: 'dashboard'
      })),
      includeReferencesDeep: true
    });
    const savedObjects = await (0, _utils.createPromiseFromStreams)([exportStream, (0, _utils.createConcatStream)([])]);
    const archive = await (0, _content.generateArchive)(params.body, (0, _content.prepareForExport)({
      savedObjects,
      source: params.path.name,
      replacedPatterns: params.body.replaced_patterns
    }));
    return response.ok({
      body: archive,
      headers: {
        'Content-Disposition': `attachment; filename="${params.body.name}-${params.body.version}.zip"`,
        'Content-Type': 'application/zip'
      }
    });
  }
});
const importContentRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /api/streams/{name}/content/import 2023-10-31',
  options: {
    access: 'public',
    summary: 'Import content into a stream',
    description: 'Links content objects to a stream.',
    body: {
      accepts: 'multipart/form-data',
      maxBytes: MAX_CONTENT_PACK_SIZE_BYTES,
      output: 'stream'
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    body: _zod.z.object({
      include: _zod.z.string().transform(value => _contentPacksSchema.contentPackIncludedObjectsSchema.parse(JSON.parse(value))),
      content: _zod.z.instanceof(_stream.Readable)
    })
  }),
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  async handler({
    params,
    request,
    getScopedClients,
    context
  }) {
    var _successResults$filte;
    const {
      assetClient,
      soClient,
      streamsClient,
      contentClient
    } = await getScopedClients({
      request
    });
    const importer = (await context.core).savedObjects.getImporter(soClient);
    await streamsClient.ensureStream(params.path.name);
    const contentPack = await (0, _content.parseArchive)(params.body.content);
    const storedContentPack = await contentClient.getStoredContentPack(params.path.name, contentPack.name).catch(err => {
      if ((0, _esErrors.isNotFoundError)(err)) {
        return {
          [_content.STREAM_NAME]: params.path.name,
          [_content.CONTENT_NAME]: contentPack.name,
          dashboards: []
        };
      }
      throw err;
    });
    const savedObjectEntries = contentPack.entries.filter(_contentPacksSchema.isSupportedSavedObjectType);
    const links = (0, _content.savedObjectLinks)(savedObjectEntries, storedContentPack);
    const savedObjects = (0, _content.prepareForImport)({
      target: params.path.name,
      include: params.body.include,
      savedObjects: savedObjectEntries,
      links
    });
    if ((0, _content.referenceManagedIndexPattern)(savedObjects)) {
      // integration package's dashboards may reference pre-existing data views
      // that we need to install before import
      await (0, _install.installManagedIndexPattern)({
        savedObjectsClient: soClient,
        savedObjectsImporter: importer
      });
    }
    const {
      successResults,
      errors = []
    } = await importer.import({
      readStream: (0, _utils.createListStream)(savedObjects),
      createNewCopies: false,
      overwrite: true
    });
    await contentClient.upsertStoredContentPack(params.path.name, {
      name: contentPack.name,
      ...links
    });
    const createdAssets = (_successResults$filte = successResults === null || successResults === void 0 ? void 0 : successResults.filter(savedObject => savedObject.type === 'dashboard').map(dashboard => ({
      [_fields.ASSET_TYPE]: 'dashboard',
      [_fields.ASSET_ID]: dashboard.id
    }))) !== null && _successResults$filte !== void 0 ? _successResults$filte : [];
    if (createdAssets.length > 0) {
      await assetClient.bulk(params.path.name, createdAssets.map(asset => ({
        index: {
          asset
        }
      })));
    }
    return {
      errors,
      created: createdAssets
    };
  }
});
const previewContentRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /internal/streams/{name}/content/preview',
  options: {
    access: 'internal',
    summary: 'Preview a content pack',
    description: 'Returns a json representation of a content pack.',
    body: {
      accepts: 'multipart/form-data',
      maxBytes: MAX_CONTENT_PACK_SIZE_BYTES,
      output: 'stream'
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    body: _zod.z.object({
      content: _zod.z.instanceof(_stream.Readable)
    })
  }),
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  async handler({
    request,
    params,
    getScopedClients
  }) {
    const {
      streamsClient
    } = await getScopedClients({
      request
    });
    await streamsClient.ensureStream(params.path.name);
    return await (0, _content.parseArchive)(params.body.content);
  }
});
const contentRoutes = exports.contentRoutes = {
  ...exportContentRoute,
  ...importContentRoute,
  ...previewContentRoute
};