"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GroupStream = void 0;
var _esErrors = require("@kbn/es-errors");
var _streamsSchema = require("@kbn/streams-schema");
var _lodash = require("lodash");
var _status_error = require("../../errors/status_error");
var _stream_active_record = require("../stream_active_record/stream_active_record");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class GroupStream extends _stream_active_record.StreamActiveRecord {
  constructor(definition, dependencies) {
    super(definition, dependencies);
  }
  clone() {
    return new GroupStream((0, _lodash.cloneDeep)(this._definition), this.dependencies);
  }
  async doHandleUpsertChange(definition, desiredState, startingState) {
    if (definition.name !== this._definition.name) {
      return {
        changeStatus: this.changeStatus,
        cascadingChanges: []
      };
    }
    if (!_streamsSchema.Streams.GroupStream.Definition.is(definition)) {
      throw new _status_error.StatusError('Cannot change stream types', 400);
    }

    // Deduplicate members
    this._definition = {
      name: definition.name,
      description: definition.description,
      group: {
        ...definition.group,
        members: Array.from(new Set(definition.group.members))
      }
    };
    return {
      cascadingChanges: [],
      changeStatus: 'upserted'
    };
  }
  async doHandleDeleteChange(target, desiredState, startingState) {
    if (target === this._definition.name) {
      return {
        cascadingChanges: [],
        changeStatus: 'deleted'
      };
    }
    // remove deleted streams from the group
    if (this.changeStatus !== 'deleted' && this._definition.group.members.includes(target)) {
      this._definition = {
        ...this._definition,
        group: {
          ...this._definition.group,
          members: this._definition.group.members.filter(member => member !== target)
        }
      };
      return {
        cascadingChanges: [],
        changeStatus: 'upserted'
      };
    }
    return {
      cascadingChanges: [],
      changeStatus: this.changeStatus
    };
  }
  async doValidateUpsertion(desiredState, startingState) {
    if (this.isDeleted()) {
      return {
        isValid: true,
        errors: []
      };
    }
    if (this._definition.name.startsWith('logs.')) {
      throw new _status_error.StatusError('A group stream name can not start with [logs.]', 400);
    }
    const existsInStartingState = startingState.has(this._definition.name);
    if (!existsInStartingState) {
      // Check for conflicts
      try {
        const dataStreamResponse = await this.dependencies.scopedClusterClient.asCurrentUser.indices.getDataStream({
          name: this._definition.name
        });
        if (dataStreamResponse.data_streams.length === 0) {
          return {
            isValid: false,
            errors: [new Error(`Cannot create group stream "${this._definition.name}" due to conflict caused by existing index`)]
          };
        }
        return {
          isValid: false,
          errors: [new Error(`Cannot create group stream "${this._definition.name}" due to conflict caused by existing data stream`)]
        };
      } catch (error) {
        if (!(0, _esErrors.isNotFoundError)(error)) {
          throw error;
        }
      }
    }

    // validate members
    for (const member of this._definition.group.members) {
      const memberStream = desiredState.get(member);
      if (!memberStream || memberStream.isDeleted()) {
        return {
          isValid: false,
          errors: [new Error(`Member stream ${member} not found`)]
        };
      }
      if (!_streamsSchema.Streams.ingest.all.Definition.is(memberStream.definition)) {
        return {
          isValid: false,
          errors: [new Error(`Member stream ${member} is neither a wired nor an unwired stream`)]
        };
      }
    }
    return {
      isValid: true,
      errors: []
    };
  }
  async doValidateDeletion(desiredState, startingState) {
    return {
      isValid: true,
      errors: []
    };
  }
  async doDetermineCreateActions() {
    return [{
      type: 'upsert_dot_streams_document',
      request: this._definition
    }];
  }
  async doDetermineUpdateActions() {
    return this.doDetermineCreateActions();
  }
  async doDetermineDeleteActions() {
    return [{
      type: 'delete_dot_streams_document',
      request: {
        name: this._definition.name
      }
    }];
  }
}
exports.GroupStream = GroupStream;