"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertToFieldDefinition = convertToFieldDefinition;
exports.filterSimulationDocuments = filterSimulationDocuments;
exports.getMappedSchemaFields = getMappedSchemaFields;
exports.getSchemaFieldsFromSimulation = getSchemaFieldsFromSimulation;
exports.getSourceFields = getSourceFields;
exports.getTableColumns = getTableColumns;
exports.getUnmappedSchemaFields = getUnmappedSchemaFields;
exports.mapField = mapField;
exports.unmapField = unmapField;
var _streamsSchema = require("@kbn/streams-schema");
var _lodash = require("lodash");
var _types = require("../../../schema_editor/types");
var _utils = require("../../../schema_editor/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getSourceFields(processors) {
  return processors.map(processor => {
    const config = (0, _streamsSchema.getProcessorConfig)(processor);
    if ('field' in config) {
      return config.field.trim();
    }
    return '';
  }).filter(Boolean);
}
function getTableColumns(processors, fields, filter) {
  const uniqueProcessorsFields = (0, _lodash.uniq)(getSourceFields(processors));
  if (filter === 'outcome_filter_failed' || filter === 'outcome_filter_skipped') {
    return uniqueProcessorsFields;
  }
  const uniqueDetectedFields = (0, _lodash.uniq)(fields.map(field => field.name));
  return (0, _lodash.uniq)([...uniqueProcessorsFields, ...uniqueDetectedFields]);
}
function filterSimulationDocuments(documents, filter) {
  switch (filter) {
    case 'outcome_filter_parsed':
      return documents.filter(doc => doc.status === 'parsed').map(doc => doc.value);
    case 'outcome_filter_partially_parsed':
      return documents.filter(doc => doc.status === 'partially_parsed').map(doc => doc.value);
    case 'outcome_filter_skipped':
      return documents.filter(doc => doc.status === 'skipped').map(doc => doc.value);
    case 'outcome_filter_failed':
      return documents.filter(doc => doc.status === 'failed').map(doc => doc.value);
    case 'outcome_filter_all':
    default:
      return documents.map(doc => doc.value);
  }
}
function getSchemaFieldsFromSimulation(detectedFields, previousDetectedFields, streamName) {
  const previousDetectedFieldsMap = previousDetectedFields.reduce((acc, field) => {
    acc[field.name] = field;
    return acc;
  }, {});
  const schemaFields = detectedFields.map(field => {
    // Detected field already mapped by the user on previous simulation
    if (previousDetectedFieldsMap[field.name]) {
      return previousDetectedFieldsMap[field.name];
    }
    // Detected field already inherited
    if ('from' in field) {
      return {
        ...field,
        status: 'inherited',
        parent: field.from
      };
    }
    // Detected field already mapped
    if ('type' in field) {
      return {
        ...field,
        status: 'mapped',
        parent: streamName
      };
    }
    // Detected field still unmapped
    return {
      status: 'unmapped',
      esType: field.esType,
      name: field.name,
      parent: streamName
    };
  });
  return schemaFields.sort(compareFieldsByStatus);
}
const statusOrder = {
  inherited: 0,
  mapped: 1,
  unmapped: 2
};
const compareFieldsByStatus = (curr, next) => {
  return statusOrder[curr.status] - statusOrder[next.status];
};
function mapField(schemaFields, updatedField) {
  return schemaFields.map(field => {
    if (field.name !== updatedField.name) return field;
    return {
      ...updatedField,
      status: 'mapped'
    };
  });
}
function unmapField(schemaFields, fieldName) {
  return schemaFields.map(field => {
    if (field.name !== fieldName) return field;
    return {
      ...field,
      status: 'unmapped'
    };
  });
}
function getMappedSchemaFields(fields) {
  return fields.filter(_types.isSchemaFieldTyped).filter(field => field.status === 'mapped');
}
function getUnmappedSchemaFields(fields) {
  return fields.filter(field => field.status === 'unmapped');
}
function convertToFieldDefinition(fields) {
  return fields.reduce((mappedFields, field) => Object.assign(mappedFields, {
    [field.name]: (0, _utils.convertToFieldDefinitionConfig)(field)
  }), {});
}