"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.StorageIndexAdapter = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _esErrors = require("@kbn/es-errors");
var _lodash = require("lodash");
var _elasticsearch = require("@elastic/elasticsearch");
var _get_schema_version = require("../get_schema_version");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function getAliasName(name) {
  return name;
}
function getBackingIndexPattern(name) {
  return `${name}-*`;
}
function getBackingIndexName(name, count) {
  const countId = (0, _lodash.padStart)(count.toString(), 6, '0');
  return `${name}-${countId}`;
}
function getIndexTemplateName(name) {
  return `${name}`;
}

// TODO: this function is here to strip properties when we add back optional/multi-value
// which should be implemented in pipelines
function toElasticsearchMappingProperty(property) {
  return property;
}
function catchConflictError(error) {
  if ((0, _esErrors.isResponseError)(error) && error.statusCode === 409) {
    return;
  }
  throw error;
}
function isNotFoundError(error) {
  return (0, _esErrors.isResponseError)(error) && error.statusCode === 404;
}

/*
 * When calling into Elasticsearch, the stack trace is lost.
 * If we create an error before calling, and append it to
 * any stack of the caught error, we get a more useful stack
 * trace.
 */
function wrapEsCall(p) {
  const error = new Error();
  return p.catch(caughtError => {
    caughtError.stack += error.stack;
    throw caughtError;
  });
}
/**
 * Adapter for writing and reading documents to/from Elasticsearch,
 * using plain indices.
 *
 * TODO:
 * - Index Lifecycle Management
 * - Schema upgrades w/ fallbacks
 */
class StorageIndexAdapter {
  constructor(esClient, logger, storage, options = {}) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "search", async request => {
      return await wrapEsCall(this.esClient.search({
        ...request,
        index: this.getSearchIndexPattern(),
        allow_no_indices: true
      }).then(response => {
        return {
          ...response,
          hits: {
            ...response.hits,
            hits: response.hits.hits.map(hit => ({
              ...hit,
              _source: this.maybeMigrateSource(hit._source)
            }))
          }
        };
      }).catch(error => {
        if (isNotFoundError(error)) {
          return {
            _shards: {
              failed: 0,
              successful: 0,
              total: 0
            },
            hits: {
              hits: [],
              total: {
                relation: 'eq',
                value: 0
              }
            },
            timed_out: false,
            took: 0
          };
        }
        throw error;
      }));
    });
    (0, _defineProperty2.default)(this, "index", async ({
      id,
      refresh = 'wait_for',
      ...request
    }) => {
      const attemptIndex = async () => {
        const [danglingItem] = id ? await this.getDanglingItems({
          ids: [id]
        }) : [undefined];
        const [indexResponse] = await Promise.all([wrapEsCall(this.esClient.index({
          ...request,
          id,
          refresh,
          index: this.getWriteTarget(),
          require_alias: true
        })), danglingItem ? wrapEsCall(this.esClient.delete({
          id: danglingItem.id,
          index: danglingItem.index,
          refresh
        })) : Promise.resolve()]);
        return indexResponse;
      };
      return this.validateComponentsBeforeWriting(attemptIndex).then(async response => {
        this.logger.debug(() => `Indexed document ${id} into ${response._index}`);
        return response;
      });
    });
    (0, _defineProperty2.default)(this, "bulk", ({
      operations,
      refresh = 'wait_for',
      ...request
    }) => {
      if (operations.length === 0) {
        this.logger.debug(`Bulk request with 0 operations is a noop`);
        return Promise.resolve({
          errors: false,
          items: [],
          took: 0,
          ingest_took: 0
        });
      }
      this.logger.debug(`Processing ${operations.length} bulk operations`);
      const bulkOperations = operations.flatMap(operation => {
        if ('index' in operation) {
          return [{
            index: {
              _id: operation.index._id
            }
          }, operation.index.document];
        }
        return [operation];
      });
      const attemptBulk = async () => {
        var _bulkOperations$flatM;
        const indexedIds = (_bulkOperations$flatM = bulkOperations.flatMap(operation => {
          if ('index' in operation && operation.index && typeof operation.index === 'object' && '_id' in operation.index && typeof operation.index._id === 'string') {
            var _operation$index$_id;
            return (_operation$index$_id = operation.index._id) !== null && _operation$index$_id !== void 0 ? _operation$index$_id : [];
          }
          return [];
        })) !== null && _bulkOperations$flatM !== void 0 ? _bulkOperations$flatM : [];
        const danglingItems = await this.getDanglingItems({
          ids: indexedIds
        });
        if (danglingItems.length) {
          this.logger.debug(`Deleting ${danglingItems.length} dangling items`);
        }
        return wrapEsCall(this.esClient.bulk({
          ...request,
          refresh,
          operations: bulkOperations.concat(danglingItems.map(item => ({
            delete: {
              _index: item.index,
              _id: item.id
            }
          }))),
          index: this.getWriteTarget(),
          require_alias: true
        }));
      };
      return this.validateComponentsBeforeWriting(attemptBulk).then(async response => {
        return response;
      });
    });
    (0, _defineProperty2.default)(this, "clean", async () => {
      const allIndices = await this.getExistingIndices();
      const hasIndices = Object.keys(allIndices).length > 0;
      // Delete all indices
      await Promise.all(Object.keys(allIndices).map(index => wrapEsCall(this.esClient.indices.delete({
        index
      }))));
      // Delete the index template
      const template = await this.getExistingIndexTemplate();
      const hasTemplate = !!template;
      if (template) {
        await wrapEsCall(this.esClient.indices.deleteIndexTemplate({
          name: getIndexTemplateName(this.storage.name)
        }));
      }
      return {
        acknowledged: true,
        result: hasIndices || hasTemplate ? 'deleted' : 'noop'
      };
    });
    (0, _defineProperty2.default)(this, "delete", async ({
      id,
      refresh = 'wait_for',
      ...request
    }) => {
      this.logger.debug(`Deleting document with id ${id}`);
      const searchResponse = await this.search({
        track_total_hits: false,
        size: 1,
        query: {
          bool: {
            filter: [{
              term: {
                _id: id
              }
            }]
          }
        }
      });
      const document = searchResponse.hits.hits[0];
      if (document) {
        await wrapEsCall(this.esClient.delete({
          ...request,
          refresh,
          id,
          index: document._index
        }));
        return {
          acknowledged: true,
          result: 'deleted'
        };
      }
      return {
        acknowledged: true,
        result: 'not_found'
      };
    });
    (0, _defineProperty2.default)(this, "get", async ({
      id,
      ...request
    }) => {
      const response = await this.search({
        track_total_hits: false,
        size: 1,
        terminate_after: 1,
        query: {
          bool: {
            filter: [{
              term: {
                _id: id
              }
            }]
          }
        },
        ...request
      });
      const hit = response.hits.hits[0];
      if (!hit) {
        throw new _elasticsearch.errors.ResponseError({
          meta: {
            aborted: false,
            attempts: 1,
            connection: null,
            context: null,
            name: 'resource_not_found_exception',
            request: {}
          },
          warnings: [],
          body: 'resource_not_found_exception',
          statusCode: 404
        });
      }
      return {
        _id: hit._id,
        _index: hit._index,
        found: true,
        _source: this.maybeMigrateSource(hit._source),
        _ignored: hit._ignored,
        _primary_term: hit._primary_term,
        _routing: hit._routing,
        _seq_no: hit._seq_no,
        _version: hit._version,
        fields: hit.fields
      };
    });
    (0, _defineProperty2.default)(this, "maybeMigrateSource", _source => {
      // check whether source is an object, if not fail
      if (typeof _source !== 'object' || _source === null) {
        throw new Error(`Source must be an object, got ${typeof _source}`);
      }
      if (this.options.migrateSource) {
        return this.options.migrateSource(_source);
      }
      return _source;
    });
    (0, _defineProperty2.default)(this, "existsIndex", () => {
      return this.esClient.indices.exists({
        index: this.getSearchIndexPattern()
      });
    });
    this.esClient = esClient;
    this.storage = storage;
    this.options = options;
    this.logger = logger.get('storage').get(this.storage.name);
  }
  getSearchIndexPattern() {
    return `${getAliasName(this.storage.name)}`;
  }
  getWriteTarget() {
    return getAliasName(this.storage.name);
  }
  async createOrUpdateIndexTemplate() {
    const version = (0, _get_schema_version.getSchemaVersion)(this.storage);
    const template = {
      mappings: {
        _meta: {
          version
        },
        dynamic: 'strict',
        properties: {
          ...(0, _lodash.mapValues)(this.storage.schema.properties, toElasticsearchMappingProperty)
        }
      },
      aliases: {
        [getAliasName(this.storage.name)]: {
          is_write_index: true
        }
      }
    };
    await wrapEsCall(this.esClient.indices.putIndexTemplate({
      name: getIndexTemplateName(this.storage.name),
      create: false,
      allow_auto_create: false,
      index_patterns: getBackingIndexPattern(this.storage.name),
      _meta: {
        version
      },
      template
    })).catch(catchConflictError);
  }
  async getExistingIndexTemplate() {
    return await wrapEsCall(this.esClient.indices.getIndexTemplate({
      name: getIndexTemplateName(this.storage.name)
    })).then(templates => {
      var _templates$index_temp;
      return (_templates$index_temp = templates.index_templates[0]) === null || _templates$index_temp === void 0 ? void 0 : _templates$index_temp.index_template;
    }).catch(error => {
      if (isNotFoundError(error)) {
        return undefined;
      }
      throw error;
    });
  }
  async getCurrentWriteIndex() {
    const [writeIndex, indices] = await Promise.all([this.getCurrentWriteIndexName(), this.getExistingIndices()]);
    return writeIndex ? {
      name: writeIndex,
      state: indices[writeIndex]
    } : undefined;
  }
  async getExistingIndices() {
    return wrapEsCall(this.esClient.indices.get({
      index: getBackingIndexPattern(this.storage.name),
      allow_no_indices: true
    }));
  }
  async getCurrentWriteIndexName() {
    const aliasName = getAliasName(this.storage.name);
    const aliases = await wrapEsCall(this.esClient.indices.getAlias({
      name: getAliasName(this.storage.name)
    })).catch(error => {
      if ((0, _esErrors.isResponseError)(error) && error.statusCode === 404) {
        return {};
      }
      throw error;
    });
    const writeIndex = Object.entries(aliases).map(([name, alias]) => {
      var _alias$aliases$aliasN;
      return {
        name,
        isWriteIndex: ((_alias$aliases$aliasN = alias.aliases[aliasName]) === null || _alias$aliases$aliasN === void 0 ? void 0 : _alias$aliases$aliasN.is_write_index) === true
      };
    }).find(({
      isWriteIndex
    }) => {
      return isWriteIndex;
    });
    return writeIndex === null || writeIndex === void 0 ? void 0 : writeIndex.name;
  }
  async createNextBackingIndex() {
    const writeIndex = await this.getCurrentWriteIndexName();
    const nextIndexName = getBackingIndexName(this.storage.name, writeIndex ? parseInt((0, _lodash.last)(writeIndex.split('-')), 10) : 1);
    await wrapEsCall(this.esClient.indices.create({
      index: nextIndexName
    })).catch(catchConflictError);
  }
  async updateMappingsOfExistingIndex({
    name
  }) {
    const simulateIndexTemplateResponse = await this.esClient.indices.simulateIndexTemplate({
      name: getBackingIndexName(this.storage.name, 999999)
    });
    if (simulateIndexTemplateResponse.template.settings) {
      await this.esClient.indices.putSettings({
        index: name,
        settings: simulateIndexTemplateResponse.template.settings
      });
    }
    if (simulateIndexTemplateResponse.template.mappings) {
      await this.esClient.indices.putMapping({
        index: name,
        ...simulateIndexTemplateResponse.template.mappings
      });
    }
  }

  /**
   * Validates whether:
   * - an index template exists
   * - the index template has the right version (if not, update it)
   * - a write index exists (if it doesn't, create it)
   * - the write index has the right version (if not, update it)
   */
  async validateComponentsBeforeWriting(cb) {
    var _existingIndexTemplat, _writeIndex$state$map, _writeIndex$state$map2;
    const [writeIndex, existingIndexTemplate] = await Promise.all([this.getCurrentWriteIndex(), this.getExistingIndexTemplate()]);
    const expectedSchemaVersion = (0, _get_schema_version.getSchemaVersion)(this.storage);
    if (!existingIndexTemplate) {
      this.logger.info(`Creating index template as it does not exist`);
      await this.createOrUpdateIndexTemplate();
    } else if (((_existingIndexTemplat = existingIndexTemplate._meta) === null || _existingIndexTemplat === void 0 ? void 0 : _existingIndexTemplat.version) !== expectedSchemaVersion) {
      this.logger.info(`Updating existing index template`);
      await this.createOrUpdateIndexTemplate();
    }
    if (!writeIndex) {
      this.logger.info(`Creating first backing index`);
      await this.createNextBackingIndex();
    } else if ((writeIndex === null || writeIndex === void 0 ? void 0 : (_writeIndex$state$map = writeIndex.state.mappings) === null || _writeIndex$state$map === void 0 ? void 0 : (_writeIndex$state$map2 = _writeIndex$state$map._meta) === null || _writeIndex$state$map2 === void 0 ? void 0 : _writeIndex$state$map2.version) !== expectedSchemaVersion) {
      this.logger.info(`Updating mappings of existing write index due to schema version mismatch`);
      await this.updateMappingsOfExistingIndex({
        name: writeIndex.name
      });
    }
    return await cb();
  }

  /**
   * Get items from all non-write indices for the specified ids.
   */
  async getDanglingItems({
    ids
  }) {
    if (!ids.length) {
      return [];
    }
    const writeIndex = await this.getCurrentWriteIndexName();
    if (writeIndex) {
      const danglingItemsResponse = await this.search({
        track_total_hits: false,
        query: {
          bool: {
            filter: [{
              terms: {
                _id: ids
              }
            }],
            must_not: [{
              term: {
                _index: writeIndex
              }
            }]
          }
        },
        size: 10_000
      });
      return danglingItemsResponse.hits.hits.map(hit => ({
        id: hit._id,
        index: hit._index
      }));
    }
    return [];
  }
  getClient() {
    return {
      bulk: this.bulk,
      delete: this.delete,
      clean: this.clean,
      index: this.index,
      search: this.search,
      get: this.get,
      existsIndex: this.existsIndex
    };
  }
}
exports.StorageIndexAdapter = StorageIndexAdapter;