"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MicrosoftDefenderEndpointConnector = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/actions-plugin/server");
var _o_auth_token_manager = require("./o_auth_token_manager");
var _constants = require("../../../common/microsoft_defender_endpoint/constants");
var _schema = require("../../../common/microsoft_defender_endpoint/schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class MicrosoftDefenderEndpointConnector extends _server.SubActionConnector {
  constructor(params) {
    super(params);
    (0, _defineProperty2.default)(this, "oAuthToken", void 0);
    (0, _defineProperty2.default)(this, "urls", void 0);
    this.oAuthToken = new _o_auth_token_manager.OAuthTokenManager({
      ...params,
      apiRequest: async (...args) => this.request(...args)
    });
    this.urls = {
      machines: `${this.config.apiUrl}/api/machines`,
      // API docs: https://learn.microsoft.com/en-us/defender-endpoint/api/get-machineactions-collection
      machineActions: `${this.config.apiUrl}/api/machineactions`,
      libraryFiles: `${this.config.apiUrl}/api/libraryfiles`
    };
    this.registerSubActions();
  }
  registerSubActions() {
    this.registerSubAction({
      name: _constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.GET_AGENT_DETAILS,
      method: 'getAgentDetails',
      schema: _schema.AgentDetailsParamsSchema
    });
    this.registerSubAction({
      name: _constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.GET_AGENT_LIST,
      method: 'getAgentList',
      schema: _schema.AgentListParamsSchema
    });
    this.registerSubAction({
      name: _constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.ISOLATE_HOST,
      method: 'isolateHost',
      schema: _schema.IsolateHostParamsSchema
    });
    this.registerSubAction({
      name: _constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.RELEASE_HOST,
      method: 'releaseHost',
      schema: _schema.ReleaseHostParamsSchema
    });
    this.registerSubAction({
      name: _constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.TEST_CONNECTOR,
      method: 'testConnector',
      schema: _schema.TestConnectorParamsSchema
    });
    this.registerSubAction({
      name: _constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.GET_ACTIONS,
      method: 'getActions',
      schema: _schema.GetActionsParamsSchema
    });
    this.registerSubAction({
      name: _constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.GET_LIBRARY_FILES,
      method: 'getLibraryFiles',
      schema: _schema.MicrosoftDefenderEndpointEmptyParamsSchema
    });
    this.registerSubAction({
      name: _constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.RUN_SCRIPT,
      method: 'runScript',
      schema: _schema.RunScriptParamsSchema
    });
    this.registerSubAction({
      name: _constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.GET_ACTION_RESULTS,
      method: 'getActionResults',
      schema: _schema.GetActionResultsParamsSchema
    });
  }
  async fetchFromMicrosoft(req, connectorUsageCollector) {
    this.logger.debug(() => `Request:\n${JSON.stringify(req, null, 2)}`);
    const requestOptions = {
      ...req,
      // We don't validate responses from Microsoft API's because we do not want failures for cases
      // where the external system might add/remove/change values in the response that we have no
      // control over.
      responseSchema: _schema.MicrosoftDefenderEndpointDoNotValidateResponseSchema,
      headers: {
        Authorization: `Bearer ${await this.oAuthToken.get(connectorUsageCollector)}`
      }
    };
    let response;
    let was401RetryDone = false;
    try {
      response = await this.request(requestOptions, connectorUsageCollector);
    } catch (err) {
      if (was401RetryDone) {
        throw err;
      }
      this.logger.debug("API call failed! Determining if it's one we can retry");

      // If error was a 401, then for some reason the token used was not valid (ex. perhaps the connector's credentials
      // were updated). IN this case, we will try again by ensuring a new token is re-generated
      if (err.message.includes('Status code: 401')) {
        this.logger.warn(`Received HTTP 401 (Unauthorized). Re-generating new access token and trying again`);
        was401RetryDone = true;
        await this.oAuthToken.generateNew(connectorUsageCollector);
        requestOptions.headers.Authorization = `Bearer ${await this.oAuthToken.get(connectorUsageCollector)}`;
        response = await this.request(requestOptions, connectorUsageCollector);
      } else {
        throw err;
      }
    }
    return response.data;
  }
  getResponseErrorMessage(error) {
    var _error$response4, _error$response5;
    const appendResponseBody = message => {
      var _error$response$data, _error$response;
      const responseBody = JSON.stringify((_error$response$data = (_error$response = error.response) === null || _error$response === void 0 ? void 0 : _error$response.data) !== null && _error$response$data !== void 0 ? _error$response$data : {});
      if (responseBody) {
        var _error$response2, _error$response2$conf, _error$response3, _error$response3$conf;
        return `${message}\nURL called:[${(_error$response2 = error.response) === null || _error$response2 === void 0 ? void 0 : (_error$response2$conf = _error$response2.config) === null || _error$response2$conf === void 0 ? void 0 : _error$response2$conf.method}] ${(_error$response3 = error.response) === null || _error$response3 === void 0 ? void 0 : (_error$response3$conf = _error$response3.config) === null || _error$response3$conf === void 0 ? void 0 : _error$response3$conf.url}\nResponse body: ${responseBody}`;
      }
      return message;
    };
    if (!((_error$response4 = error.response) !== null && _error$response4 !== void 0 && _error$response4.status)) {
      var _error$message;
      return appendResponseBody((_error$message = error.message) !== null && _error$message !== void 0 ? _error$message : 'Unknown API Error');
    }
    if (error.response.status === 401) {
      return appendResponseBody('Unauthorized API Error (401)');
    }
    return appendResponseBody(`API Error: [${(_error$response5 = error.response) === null || _error$response5 === void 0 ? void 0 : _error$response5.statusText}] ${error.message}`);
  }
  buildODataUrlParams({
    filter = {},
    page = 1,
    pageSize = 20,
    sortField = '',
    sortDirection = 'desc'
  }) {
    const oDataQueryOptions = {
      $count: true
    };
    if (pageSize) {
      oDataQueryOptions.$top = pageSize;
    }
    if (page > 1) {
      oDataQueryOptions.$skip = page * pageSize - pageSize;
    }
    if (sortField) {
      oDataQueryOptions.$orderby = `${sortField} ${sortDirection}`;
    }
    const filterEntries = Object.entries(filter);
    if (filterEntries.length > 0) {
      oDataQueryOptions.$filter = '';
      for (const [key, value] of filterEntries) {
        const isArrayValue = Array.isArray(value);
        if (oDataQueryOptions.$filter) {
          oDataQueryOptions.$filter += ' AND ';
        }
        oDataQueryOptions.$filter += `${key} ${isArrayValue ? 'in' : 'eq'} ${isArrayValue ? '(' + value.map(valueString => `'${valueString}'`).join(',') + ')' : `'${value}'`}`;
      }
    }
    return oDataQueryOptions;
  }
  async testConnector(_, connectorUsageCollector) {
    const results = [];
    const catchErrorAndIgnoreExpectedErrors = err => {
      if (err.message.includes('ResourceNotFound')) {
        return '';
      }
      throw err;
    };
    await this.getAgentDetails({
      id: 'elastic-connector-test'
    }, connectorUsageCollector).catch(catchErrorAndIgnoreExpectedErrors).then(() => {
      results.push('API call to Machines API was successful');
    });
    await this.isolateHost({
      id: 'elastic-connector-test',
      comment: 'connector test'
    }, connectorUsageCollector).catch(catchErrorAndIgnoreExpectedErrors).then(() => {
      results.push('API call to Machine Isolate was successful');
    });
    await this.releaseHost({
      id: 'elastic-connector-test',
      comment: 'connector test'
    }, connectorUsageCollector).catch(catchErrorAndIgnoreExpectedErrors).then(() => {
      results.push('API call to Machine Release was successful');
    });
    await this.getActions({
      pageSize: 1
    }, connectorUsageCollector).catch(catchErrorAndIgnoreExpectedErrors).then(() => {
      results.push('API call to Machine Actions was successful');
    });
    await this.runScript({
      id: 'elastic-connector-test',
      comment: 'connector test',
      parameters: {
        scriptName: 'test'
      }
    }, connectorUsageCollector).catch(catchErrorAndIgnoreExpectedErrors).then(() => {
      results.push('API call to Machine RunScript was successful');
    });
    return {
      results
    };
  }
  async getAgentDetails({
    id
  }, connectorUsageCollector) {
    // API Reference: https://learn.microsoft.com/en-us/defender-endpoint/api/machine

    return this.fetchFromMicrosoft({
      url: `${this.urls.machines}/${id}`
    }, connectorUsageCollector);
  }
  async getAgentList({
    page = 1,
    pageSize = 20,
    ...filter
  }, connectorUsageCollector) {
    var _response$OdataCoun;
    // API Reference: https://learn.microsoft.com/en-us/defender-endpoint/api/get-machines
    // OData usage reference: https://learn.microsoft.com/en-us/defender-endpoint/api/exposed-apis-odata-samples

    const response = await this.fetchFromMicrosoft({
      url: `${this.urls.machines}`,
      method: 'GET',
      params: this.buildODataUrlParams({
        filter,
        page,
        pageSize
      })
    }, connectorUsageCollector);
    return {
      ...response,
      page,
      pageSize,
      total: (_response$OdataCoun = response['@odata.count']) !== null && _response$OdataCoun !== void 0 ? _response$OdataCoun : -1
    };
  }
  async isolateHost({
    id,
    comment
  }, connectorUsageCollector) {
    // API Reference: https://learn.microsoft.com/en-us/defender-endpoint/api/isolate-machine

    return this.fetchFromMicrosoft({
      url: `${this.urls.machines}/${id}/isolate`,
      method: 'POST',
      data: {
        Comment: comment,
        IsolationType: 'Full'
      }
    }, connectorUsageCollector);
  }
  async releaseHost({
    id,
    comment
  }, connectorUsageCollector) {
    // API Reference:https://learn.microsoft.com/en-us/defender-endpoint/api/unisolate-machine

    return this.fetchFromMicrosoft({
      url: `${this.urls.machines}/${id}/unisolate`,
      method: 'POST',
      data: {
        Comment: comment
      }
    }, connectorUsageCollector);
  }
  async runScript(payload, connectorUsageCollector) {
    // API Reference:https://learn.microsoft.com/en-us/defender-endpoint/api/run-live-response

    return this.fetchFromMicrosoft({
      url: `${this.urls.machines}/${payload.id}/runliveresponse`,
      method: 'POST',
      data: {
        Comment: payload.comment,
        Commands: [{
          type: 'RunScript',
          params: [{
            key: 'ScriptName',
            value: payload.parameters.scriptName
          }, {
            key: 'Args',
            value: payload.parameters.args || '--noargs'
          }]
        }]
      }
    }, connectorUsageCollector);
  }
  async getActions({
    page = 1,
    pageSize = 20,
    sortField,
    sortDirection = 'desc',
    ...filter
  }, connectorUsageCollector) {
    var _response$OdataCoun2;
    // API Reference: https://learn.microsoft.com/en-us/defender-endpoint/api/get-machineactions-collection
    // OData usage reference: https://learn.microsoft.com/en-us/defender-endpoint/api/exposed-apis-odata-samples

    const response = await this.fetchFromMicrosoft({
      url: `${this.urls.machineActions}`,
      method: 'GET',
      params: this.buildODataUrlParams({
        filter,
        page,
        pageSize,
        sortField,
        sortDirection
      })
    }, connectorUsageCollector);
    return {
      ...response,
      page,
      pageSize,
      total: (_response$OdataCoun2 = response['@odata.count']) !== null && _response$OdataCoun2 !== void 0 ? _response$OdataCoun2 : -1
    };
  }
  async getActionResults({
    id
  }, connectorUsageCollector) {
    // API Reference: https://learn.microsoft.com/en-us/defender-endpoint/api/get-live-response-result

    const resultDownloadLink = await this.fetchFromMicrosoft({
      url: `${this.urls.machineActions}/${id}/GetLiveResponseResultDownloadLink(index=0)`,
      // We want to download the first result
      method: 'GET'
    }, connectorUsageCollector);
    this.logger.debug(() => `script results for machineId [${id}]:\n${JSON.stringify(resultDownloadLink)}`);
    const fileUrl = resultDownloadLink.value;
    if (!fileUrl) {
      throw new Error(`Download URL for script results of machineId [${id}] not found`);
    }
    const downloadConnection = await this.request({
      url: fileUrl,
      method: 'get',
      responseType: 'stream',
      responseSchema: _schema.DownloadActionResultsResponseSchema
    }, connectorUsageCollector);
    return downloadConnection.data;
  }
  async getLibraryFiles(payload, connectorUsageCollector) {
    // API Reference:https://learn.microsoft.com/en-us/defender-endpoint/api/list-library-files

    return this.fetchFromMicrosoft({
      url: `${this.urls.libraryFiles}`,
      method: 'GET'
    }, connectorUsageCollector);
  }
}
exports.MicrosoftDefenderEndpointConnector = MicrosoftDefenderEndpointConnector;