"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.throwIfSubActionIsNotSupported = exports.throwIfAdditionalFieldsNotSupported = exports.prepareIncident = exports.getPushedDate = exports.getAxiosInstance = exports.createServiceError = void 0;
var _axios = _interopRequireWildcard(require("axios"));
var _axios_utils = require("@kbn/actions-plugin/server/lib/axios_utils");
var _get_oauth_jwt_access_token = require("@kbn/actions-plugin/server/lib/get_oauth_jwt_access_token");
var _server = require("@kbn/actions-plugin/server");
var _config = require("./config");
var i18n = _interopRequireWildcard(require("./translations"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const prepareIncident = (useOldApi, incident) => {
  const {
    additional_fields: additionalFields,
    ...restIncidentFields
  } = incident;
  if (useOldApi) {
    return restIncidentFields;
  }
  const baseFields = Object.entries(restIncidentFields).reduce((acc, [key, value]) => ({
    ...acc,
    [`${_config.FIELD_PREFIX}${key}`]: value
  }), {});
  return {
    ...additionalFields,
    ...baseFields
  };
};
exports.prepareIncident = prepareIncident;
const createErrorMessage = errorResponse => {
  if (errorResponse == null) {
    return 'unknown: errorResponse was null';
  }
  const {
    error
  } = errorResponse;
  return error != null ? `${error === null || error === void 0 ? void 0 : error.message}: ${error === null || error === void 0 ? void 0 : error.detail}` : 'unknown: no error in error response';
};
const createServiceError = (error, message) => {
  var _error$response;
  const serviceError = new _axios.AxiosError((0, _axios_utils.getErrorMessage)(i18n.SERVICENOW, `${message}. Error: ${error.message} Reason: ${createErrorMessage((_error$response = error.response) === null || _error$response === void 0 ? void 0 : _error$response.data)}`));
  serviceError.code = error.code;
  serviceError.config = error.config;
  serviceError.request = error.request;
  serviceError.response = error.response;
  return serviceError;
};
exports.createServiceError = createServiceError;
const getPushedDate = timestamp => {
  if (timestamp != null) {
    return new Date((0, _axios_utils.addTimeZoneToDate)(timestamp)).toISOString();
  }
  return new Date().toISOString();
};
exports.getPushedDate = getPushedDate;
const throwIfSubActionIsNotSupported = ({
  api,
  subAction,
  supportedSubActions,
  logger
}) => {
  if (!api[subAction]) {
    const errorMessage = `[Action][ExternalService] Unsupported subAction type ${subAction}.`;
    logger.error(errorMessage);
    throw new Error(errorMessage);
  }
  if (!supportedSubActions.includes(subAction)) {
    const errorMessage = `[Action][ExternalService] subAction ${subAction} not implemented.`;
    logger.error(errorMessage);
    throw new Error(errorMessage);
  }
};
exports.throwIfSubActionIsNotSupported = throwIfSubActionIsNotSupported;
const throwIfAdditionalFieldsNotSupported = (useOldApi, incident) => {
  if (useOldApi && incident.additional_fields) {
    throw new _axios.AxiosError('ServiceNow additional fields are not supported for deprecated connectors.', '400');
  }
};
exports.throwIfAdditionalFieldsNotSupported = throwIfAdditionalFieldsNotSupported;
const getAxiosInstance = ({
  connectorId,
  logger,
  configurationUtilities,
  credentials,
  snServiceUrl,
  connectorTokenClient
}) => {
  const {
    config,
    secrets
  } = credentials;
  const {
    isOAuth
  } = config;
  const {
    username,
    password
  } = secrets;
  let axiosInstance;
  if (!isOAuth && username && password) {
    axiosInstance = _axios.default.create({
      headers: (0, _server.getBasicAuthHeader)({
        username,
        password
      })
    });
  } else {
    axiosInstance = _axios.default.create();
    axiosInstance.interceptors.request.use(async axiosConfig => {
      const accessToken = await (0, _get_oauth_jwt_access_token.getOAuthJwtAccessToken)({
        connectorId,
        logger,
        configurationUtilities,
        credentials: {
          config: {
            clientId: config.clientId,
            jwtKeyId: config.jwtKeyId,
            userIdentifierValue: config.userIdentifierValue
          },
          secrets: {
            clientSecret: secrets.clientSecret,
            privateKey: secrets.privateKey,
            privateKeyPassword: secrets.privateKeyPassword ? secrets.privateKeyPassword : null
          }
        },
        tokenUrl: `${snServiceUrl}/oauth_token.do`,
        connectorTokenClient
      });
      if (!accessToken) {
        throw new Error(`Unable to retrieve access token for connectorId: ${connectorId}`);
      }
      axiosConfig.headers = new _axios.AxiosHeaders({
        ...axiosConfig.headers,
        Authorization: accessToken
      });
      return axiosConfig;
    }, error => {
      return Promise.reject(error);
    });
    axiosInstance.interceptors.response.use(response => response, async error => {
      var _error$response2;
      const statusCode = error === null || error === void 0 ? void 0 : (_error$response2 = error.response) === null || _error$response2 === void 0 ? void 0 : _error$response2.status;

      // Look for 4xx errors that indicate something is wrong with the request
      // We don't know for sure that it is an access token issue but remove saved
      // token just to be sure
      if (statusCode >= 400 && statusCode < 500) {
        await connectorTokenClient.deleteConnectorTokens({
          connectorId
        });
      }
      return Promise.reject(error);
    });
  }
  return axiosInstance;
};
exports.getAxiosInstance = getAxiosInstance;