"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initSetSolutionSpaceApi = initSetSolutionSpaceApi;
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/core/server");
var _errors = require("../../../lib/errors");
var _space_schema = require("../../../lib/space_schema");
var _lib = require("../../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const spaceSolutionSchema = _configSchema.schema.oneOf([_configSchema.schema.object({
  solution: _space_schema.solutionSchema
}), _configSchema.schema.object({
  solution_type: _configSchema.schema.oneOf([_configSchema.schema.literal('security'), _configSchema.schema.literal('observability'), _configSchema.schema.literal('elasticsearch'), _configSchema.schema.literal('search')])
})]);

/* FUTURE Engineer
 * This route /internal/spaces/space/{id}/solution is and will be used by cloud (control panel)
 * to set the solution of a default space for an instant deployment
 * and it will use the parameter "solution_type"
 */

function initSetSolutionSpaceApi(deps) {
  const {
    router,
    getSpacesService
  } = deps;
  router.put({
    path: '/internal/spaces/space/{id}/solution',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the spaces service via a scoped spaces client'
      }
    },
    options: {
      description: `Update solution for a space`
    },
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      }),
      body: spaceSolutionSchema
    }
  }, (0, _lib.createLicensedRouteHandler)(async (context, request, response) => {
    const spacesClient = (await getSpacesService()).createSpacesClient(request);
    const id = request.params.id;
    let solutionToUpdate;
    let result;
    try {
      if ('solution' in request.body) {
        solutionToUpdate = request.body.solution;
      } else {
        solutionToUpdate = (0, _lib.parseCloudSolution)(request.body.solution_type);
      }
      const space = await (spacesClient === null || spacesClient === void 0 ? void 0 : spacesClient.get(id));
      result = await spacesClient.update(id, {
        ...space,
        solution: solutionToUpdate
      });
    } catch (error) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
        return response.notFound();
      }
      return response.customError((0, _errors.wrapError)(error));
    }
    return response.ok({
      body: result
    });
  }));
}