"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initGetAllSpacesApi = initGetAllSpacesApi;
var _configSchema = require("@kbn/config-schema");
var _common = require("../../../../common");
var _errors = require("../../../lib/errors");
var _lib = require("../../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function initGetAllSpacesApi(deps) {
  const {
    router,
    log,
    getSpacesService
  } = deps;
  router.versioned.get({
    path: '/api/spaces/space',
    access: 'public',
    summary: `Get all spaces`,
    options: {
      tags: ['oas-tag:spaces']
    }
  }).addVersion({
    version: _common.API_VERSIONS.public.v1,
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the spaces service via a scoped spaces client'
      }
    },
    validate: {
      request: {
        query: _configSchema.schema.object({
          purpose: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('any'), _configSchema.schema.literal('copySavedObjectsIntoSpace'), _configSchema.schema.literal('shareSavedObjectsIntoSpace')], {
            meta: {
              description: 'Specifies which authorization checks are applied to the API call. The default value is `any`.'
            }
          })),
          include_authorized_purposes: _configSchema.schema.conditional(_configSchema.schema.siblingRef('purpose'), _configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.literal(false)), _configSchema.schema.maybe(_configSchema.schema.boolean()), {
            meta: {
              description: 'When enabled, the API returns any spaces that the user is authorized to access in any capacity and each space will contain the purposes for which the user is authorized. This can be useful to determine which spaces a user can read but not take a specific action in. If the security plugin is not enabled, this parameter has no effect, since no authorization checks take place. This parameter cannot be used in with the `purpose` parameter.'
            }
          })
        })
      },
      response: {
        200: {
          description: 'Indicates a successful call.'
        }
      }
    }
  }, (0, _lib.createLicensedRouteHandler)(async (context, request, response) => {
    log.debug(`Inside GET /api/spaces/space`);
    const {
      purpose,
      include_authorized_purposes: includeAuthorizedPurposes
    } = request.query;
    const spacesClient = getSpacesService().createSpacesClient(request);
    let spaces;
    try {
      log.debug(`Attempting to retrieve all spaces for ${purpose} purpose with includeAuthorizedPurposes=${includeAuthorizedPurposes}`);
      spaces = await spacesClient.getAll({
        purpose,
        includeAuthorizedPurposes
      });
      log.debug(`Retrieved ${spaces.length} spaces for ${purpose} purpose with includeAuthorizedPurposes=${includeAuthorizedPurposes}`);
    } catch (error) {
      log.debug(`Error retrieving spaces for ${purpose} purpose with includeAuthorizedPurposes=${includeAuthorizedPurposes}: ${error}`);
      return response.customError((0, _errors.wrapError)(error));
    }
    return response.ok({
      body: spaces
    });
  }));
}