"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DefaultResourceInstaller = void 0;
var _constants = require("../../common/constants");
var _sli_mappings_template = require("../assets/component_templates/sli_mappings_template");
var _sli_settings_template = require("../assets/component_templates/sli_settings_template");
var _summary_mappings_template = require("../assets/component_templates/summary_mappings_template");
var _summary_settings_template = require("../assets/component_templates/summary_settings_template");
var _sli_index_template = require("../assets/index_templates/sli_index_template");
var _summary_index_template = require("../assets/index_templates/summary_index_template");
var _retry = require("../utils/retry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DefaultResourceInstaller {
  constructor(esClient, logger) {
    this.esClient = esClient;
    this.logger = logger;
  }
  async ensureCommonResourcesInstalled() {
    try {
      this.logger.debug('Installing SLO shared resources');
      await Promise.all([this.createOrUpdateComponentTemplate(_sli_mappings_template.SLI_MAPPINGS_TEMPLATE), this.createOrUpdateComponentTemplate(_sli_settings_template.SLI_SETTINGS_TEMPLATE), this.createOrUpdateComponentTemplate(_summary_mappings_template.SUMMARY_MAPPINGS_TEMPLATE), this.createOrUpdateComponentTemplate(_summary_settings_template.SUMMARY_SETTINGS_TEMPLATE)]);
      await this.createOrUpdateIndexTemplate(_sli_index_template.SLI_INDEX_TEMPLATE);
      await this.createOrUpdateIndexTemplate(_summary_index_template.SUMMARY_INDEX_TEMPLATE);
      await this.createIndex(_constants.SLI_DESTINATION_INDEX_NAME);
      await this.createIndex(_constants.SUMMARY_DESTINATION_INDEX_NAME);
      await this.createIndex(_constants.SUMMARY_TEMP_INDEX_NAME);
    } catch (err) {
      this.logger.error(`Error while installing SLO shared resources: ${err}`);
    }
  }
  async createOrUpdateComponentTemplate(template) {
    var _template$_meta;
    const currentVersion = await this.fetchComponentTemplateVersion(template.name);
    if ((_template$_meta = template._meta) !== null && _template$_meta !== void 0 && _template$_meta.version && currentVersion === template._meta.version) {
      this.logger.debug(`SLO component template found with version [${template._meta.version}]`);
    } else {
      this.logger.debug(`Installing SLO component template [${template.name}]`);
      return this.execute(() => this.esClient.cluster.putComponentTemplate(template));
    }
  }
  async createOrUpdateIndexTemplate(template) {
    var _template$_meta2;
    const currentVersion = await this.fetchIndexTemplateVersion(template.name);
    if ((_template$_meta2 = template._meta) !== null && _template$_meta2 !== void 0 && _template$_meta2.version && currentVersion === template._meta.version) {
      this.logger.debug(`SLO index template found with version [${template._meta.version}]`);
    } else {
      this.logger.debug(`Installing SLO index template [${template.name}]`);
      return this.execute(() => this.esClient.indices.putIndexTemplate(template));
    }
  }
  async createIndex(indexName) {
    try {
      await this.execute(() => this.esClient.indices.create({
        index: indexName
      }));
    } catch (err) {
      var _err$meta, _err$meta$body, _err$meta$body$error;
      if ((err === null || err === void 0 ? void 0 : (_err$meta = err.meta) === null || _err$meta === void 0 ? void 0 : (_err$meta$body = _err$meta.body) === null || _err$meta$body === void 0 ? void 0 : (_err$meta$body$error = _err$meta$body.error) === null || _err$meta$body$error === void 0 ? void 0 : _err$meta$body$error.type) !== 'resource_already_exists_exception') {
        throw err;
      }
    }
  }
  async execute(esCall) {
    return await (0, _retry.retryTransientEsErrors)(esCall, {
      logger: this.logger
    });
  }
  async fetchComponentTemplateVersion(name) {
    var _getTemplateRes$compo, _getTemplateRes$compo2, _getTemplateRes$compo3, _getTemplateRes$compo4, _getTemplateRes$compo5;
    const getTemplateRes = await this.execute(() => this.esClient.cluster.getComponentTemplate({
      name
    }, {
      ignore: [404]
    }));
    return (_getTemplateRes$compo = getTemplateRes === null || getTemplateRes === void 0 ? void 0 : (_getTemplateRes$compo2 = getTemplateRes.component_templates) === null || _getTemplateRes$compo2 === void 0 ? void 0 : (_getTemplateRes$compo3 = _getTemplateRes$compo2[0]) === null || _getTemplateRes$compo3 === void 0 ? void 0 : (_getTemplateRes$compo4 = _getTemplateRes$compo3.component_template) === null || _getTemplateRes$compo4 === void 0 ? void 0 : (_getTemplateRes$compo5 = _getTemplateRes$compo4._meta) === null || _getTemplateRes$compo5 === void 0 ? void 0 : _getTemplateRes$compo5.version) !== null && _getTemplateRes$compo !== void 0 ? _getTemplateRes$compo : null;
  }
  async fetchIndexTemplateVersion(name) {
    var _getTemplateRes$index, _getTemplateRes$index2, _getTemplateRes$index3, _getTemplateRes$index4, _getTemplateRes$index5;
    const getTemplateRes = await this.execute(() => this.esClient.indices.getIndexTemplate({
      name
    }, {
      ignore: [404]
    }));
    return (_getTemplateRes$index = getTemplateRes === null || getTemplateRes === void 0 ? void 0 : (_getTemplateRes$index2 = getTemplateRes.index_templates) === null || _getTemplateRes$index2 === void 0 ? void 0 : (_getTemplateRes$index3 = _getTemplateRes$index2[0]) === null || _getTemplateRes$index3 === void 0 ? void 0 : (_getTemplateRes$index4 = _getTemplateRes$index3.index_template) === null || _getTemplateRes$index4 === void 0 ? void 0 : (_getTemplateRes$index5 = _getTemplateRes$index4._meta) === null || _getTemplateRes$index5 === void 0 ? void 0 : _getTemplateRes$index5.version) !== null && _getTemplateRes$index !== void 0 ? _getTemplateRes$index : null;
  }
}
exports.DefaultResourceInstaller = DefaultResourceInstaller;