"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getUsageRecords = exports.getCloudDefendUsageRecords = void 0;
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const BATCH_SIZE = 1000;
const SAMPLE_WEIGHT_SECONDS = 3600; // 1 Hour

const buildMeteringRecord = (agentId, timestampStr, taskId, tier, projectId) => {
  const timestamp = new Date(timestampStr);
  timestamp.setMinutes(0);
  timestamp.setSeconds(0);
  timestamp.setMilliseconds(0);
  const creationTimestamp = new Date();
  const usageRecord = {
    id: `${projectId}_${agentId}_${timestamp.toISOString()}`,
    usage_timestamp: timestampStr,
    creation_timestamp: creationTimestamp.toISOString(),
    usage: {
      type: _constants.CLOUD_SECURITY_TASK_TYPE,
      sub_type: _constants.CLOUD_DEFEND,
      period_seconds: SAMPLE_WEIGHT_SECONDS,
      quantity: 1
    },
    source: {
      id: taskId,
      instance_group_id: projectId,
      metadata: {
        tier
      }
    }
  };
  return usageRecord;
};
const getUsageRecords = async (esClient, searchFrom, searchAfter) => {
  return esClient.search({
    index: _constants.CLOUD_DEFEND_HEARTBEAT_INDEX,
    size: BATCH_SIZE,
    sort: [{
      'event.ingested': 'asc'
    }, {
      'agent.id': 'asc'
    }],
    search_after: searchAfter,
    query: {
      bool: {
        must: [{
          range: {
            'event.ingested': {
              //  gt: searchFrom.toISOString(), Tech debt: https://github.com/elastic/security-team/issues/9895
              gte: `now-30m`
            }
          }
        }, {
          term: {
            'cloud_defend.block_action_enabled': true
          }
        }]
      }
    }
  }, {
    ignore: [404]
  });
};
exports.getUsageRecords = getUsageRecords;
const getCloudDefendUsageRecords = async ({
  esClient,
  projectId,
  taskId,
  lastSuccessfulReport,
  cloudSecuritySolution,
  tier,
  logger
}) => {
  try {
    let allRecords = [];
    let searchAfter;
    let fetchMore = true;
    while (fetchMore) {
      var _usageRecords$hits, _usageRecords$hits$hi;
      const usageRecords = await getUsageRecords(esClient, lastSuccessfulReport, searchAfter);
      if (!(usageRecords !== null && usageRecords !== void 0 && (_usageRecords$hits = usageRecords.hits) !== null && _usageRecords$hits !== void 0 && (_usageRecords$hits$hi = _usageRecords$hits.hits) !== null && _usageRecords$hits$hi !== void 0 && _usageRecords$hits$hi.length)) {
        break;
      }
      const records = usageRecords.hits.hits.reduce((acc, {
        _source
      }) => {
        if (!_source) {
          return acc;
        }
        const {
          event
        } = _source;
        const record = buildMeteringRecord(_source['agent.id'], event.ingested, taskId, tier, projectId);
        return [...acc, record];
      }, []);
      allRecords = [...allRecords, ...records];
      if (usageRecords.hits.hits.length < BATCH_SIZE) {
        fetchMore = false;
      } else {
        searchAfter = usageRecords.hits.hits[usageRecords.hits.hits.length - 1].sort;
      }
    }
    return allRecords;
  } catch (err) {
    logger.error(`Failed to fetch ${cloudSecuritySolution} metering data ${err}`);
  }
};
exports.getCloudDefendUsageRecords = getCloudDefendUsageRecords;