"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildTopNFlowQuery = exports.buildTopNFlowCountQuery = void 0;
var _build_query = require("../../../../../utils/build_query");
var _helpers = require("../helpers");
var _helpers2 = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const buildTopNFlowQuery = ({
  defaultIndex,
  filterQuery,
  flowTarget,
  sort,
  pagination,
  timerange,
  ip
}) => {
  var _pagination$querySize;
  const querySize = (_pagination$querySize = pagination === null || pagination === void 0 ? void 0 : pagination.querySize) !== null && _pagination$querySize !== void 0 ? _pagination$querySize : 10;
  const query = getQuery({
    filterQuery,
    flowTarget,
    timerange,
    ip
  });
  const dslQuery = {
    allow_no_indices: true,
    index: defaultIndex,
    ignore_unavailable: true,
    body: {
      aggregations: getFlowTargetAggs(sort, flowTarget, querySize),
      query,
      _source: false,
      fields: [{
        field: '@timestamp',
        format: 'strict_date_optional_time'
      }],
      size: 0
    },
    track_total_hits: false
  };
  return dslQuery;
};
exports.buildTopNFlowQuery = buildTopNFlowQuery;
const buildTopNFlowCountQuery = ({
  defaultIndex,
  filterQuery,
  flowTarget,
  timerange,
  ip
}) => {
  const query = getQuery({
    filterQuery,
    flowTarget,
    timerange,
    ip
  });
  const dslQuery = {
    allow_no_indices: true,
    index: defaultIndex,
    ignore_unavailable: true,
    body: {
      aggregations: getCountAgg(flowTarget),
      query,
      _source: false,
      size: 0
    },
    track_total_hits: false
  };
  return dslQuery;
};

// creates the dsl bool query with the filters
exports.buildTopNFlowCountQuery = buildTopNFlowCountQuery;
const getQuery = ({
  filterQuery,
  flowTarget,
  timerange: {
    from,
    to
  },
  ip
}) => ({
  bool: {
    filter: [...(0, _build_query.createQueryFilterClauses)(filterQuery), getTimeRangeFilter(from, to)],
    ...(ip && {
      should: [{
        term: {
          [`${(0, _helpers.getOppositeField)(flowTarget)}.ip`]: ip
        }
      }],
      minimum_should_match: 1
    })
  }
});
const getTimeRangeFilter = (from, to) => ({
  range: {
    '@timestamp': {
      gte: from,
      lte: to,
      format: 'strict_date_optional_time'
    }
  }
});
const getCountAgg = flowTarget => ({
  top_n_flow_count: {
    cardinality: {
      field: `${flowTarget}.ip`
    }
  }
});
const getFlowTargetAggs = (sort, flowTarget, querySize) => ({
  [flowTarget]: {
    terms: {
      field: `${flowTarget}.ip`,
      size: querySize,
      order: {
        ...(0, _helpers2.getQueryOrder)(sort)
      }
    },
    aggs: {
      bytes_in: {
        sum: {
          field: `${(0, _helpers.getOppositeField)(flowTarget)}.bytes`
        }
      },
      bytes_out: {
        sum: {
          field: `${flowTarget}.bytes`
        }
      },
      domain: {
        terms: {
          field: `${flowTarget}.domain`,
          order: {
            timestamp: 'desc'
          }
        },
        aggs: {
          timestamp: {
            max: {
              field: '@timestamp'
            }
          }
        }
      },
      location: {
        filter: {
          exists: {
            field: `${flowTarget}.geo`
          }
        },
        aggs: {
          top_geo: {
            top_hits: {
              _source: false,
              fields: [`${flowTarget}.geo.*`, {
                field: '@timestamp',
                format: 'strict_date_optional_time'
              }],
              size: 1
            }
          }
        }
      },
      autonomous_system: {
        filter: {
          exists: {
            field: `${flowTarget}.as`
          }
        },
        aggs: {
          top_as: {
            top_hits: {
              _source: false,
              fields: [`${flowTarget}.as.*`, {
                field: '@timestamp',
                format: 'strict_date_optional_time'
              }],
              size: 1
            }
          }
        }
      },
      flows: {
        cardinality: {
          field: 'network.community_id'
        }
      },
      [`${(0, _helpers.getOppositeField)(flowTarget)}_ips`]: {
        cardinality: {
          field: `${(0, _helpers.getOppositeField)(flowTarget)}.ip`
        }
      }
    }
  }
});